<?php declare(strict_types = 1);

namespace Core\Components\Navigation;

use Core\Model\Providers\ISiteMapUrl;
use DateTimeInterface;
use Navigations\Model\Entities\Navigation;
use Nette\SmartObject;
use Nette\Utils\ArrayHash;

class DaoNavigationItem implements ISiteMapUrl
{
	use SmartObject;

	/** @var int */
	public $id;

	/** @var string */
	public $title;

	/** @var string */
	public $alias;

	public bool $hidden = false;

	/** @var bool|int */
	public $isHomepage = false;

	/** @var string */
	public $lang;

	/** @var array */
	public $seo;

	/** @var int */
	public $lvl;

	/** @var string|array */
	public $link;

	public bool $isFullLinkFilled = false;

	public ?string $target = null;

	/** @var string|null */
	public ?string $ico = null;

	/** @var string|null */
	public ?string $icoSecondary = null;

	/** @var string */
	public $icoClass;

	/** @var string */
	public $img;

	/** @var string */
	public $description;

	/** @var bool */
	public $isParent = false;

	/** @var string */
	public $class = '';

	/** @var bool */
	public $isCurrent = false;

	/** @var array */
	public $childs = [];

	public ?DaoNavigationItem $parent   = null;
	public ?int               $parentId = 0;

	/** @var int */
	public $order = 100;

	/** @var string */
	public $openIn;

	/** @var string|null */
	public ?string $customText1 = null;

	/** @var string|null */
	public ?string $customText2 = null;

	/** @var bool|int */
	public $hideInMobileMenu = false;

	/** @var bool|int */
	public $hideInDesktopMenu = false;

	protected ?DateTimeInterface $modified        = null;
	public array                 $security        = [];
	public bool                  $securityConfig  = true;
	public ?string               $componentType   = null;
	public array                 $componentParams = [];
	protected array              $params          = [];
	public ?string               $groupType       = null;
	public ?string               $confirm         = null;

	public array $dataAttrs = [];

	/**
	 * @param array|ArrayHash $arr
	 */
	public function __construct($arr = [], ?DaoNavigationItem $parent = null)
	{
		$parameters = get_class_vars(self::class);

		if (is_object($arr)) {
			$tmp = (array) $arr;
			if ($arr instanceof Navigation) {
				foreach ($arr->getParams() as $k => $v) {
					$tmp[$k] = $v;
				}
			}

			$arr = $tmp;
		}

		foreach ($arr as $key => $value) {
			if (strpos($key, 'data-') === 0) {
				$this->dataAttrs[$key] = $value;
			}

			if ($key === 'ico' && strlen((string) $value) > 3) {
				$expl  = explode(' ', $value);
				$value = count($expl) === 1 ? 'fa fa-' . $value : $value;
			}

			if (array_key_exists($key, $parameters)) {
				$this->$key = $value;
			} else if ($key === 'items') {
				$this->isParent = true;

				foreach ($value as $deepK => $deepV) {
					$this->childs[$deepK] = new DaoNavigationItem($deepV, $this);
				}
			}
		}

		$this->parent = &$parent;
	}

	public function getId(): ?int { return $this->id; }

	public function isChildrenCurrent(): bool
	{
		foreach ($this->childs as $item) {
			if ($item->isCurrent) {
				return true;
			}
		}

		return false;
	}

	public function addChild(DaoNavigationItem $dao): self
	{
		$this->childs[] = $dao;

		return $this;
	}

	public function setParams(array $arr): self
	{
		$this->params = $arr;

		return $this;
	}

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		$this->params[$key] = $value;

		return $this;
	}

	/**
	 * @return mixed|null
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

	public function getLang(): ?string { return $this->lang; }

	/**
	 * @param string|array $key
	 * @param mixed|null   $value
	 */
	public function setSeo($key, $value = null): self
	{
		if (is_array($key)) {
			$this->seo = $key;
		} else {
			$this->seo[$key] = $value;
		}

		return $this;
	}

	/**
	 * @return array|mixed|null
	 */
	public function getSeo(?string $key = null)
	{
		if ($key) {
			return $this->seo[$key] ?? null;
		}

		return $this->seo ?: [];
	}

	public function setModified(DateTimeInterface $dateTime): self
	{
		$this->modified = $dateTime;

		return $this;
	}

	public function getModified(): ?DateTimeInterface { return $this->modified; }

	public function getImages(): array { return []; }

	public function renderDataAttrs(): string
	{
		$arr = [];

		foreach ($this->dataAttrs as $k => $v) {
			$arr[] = "$k=\"$v\"";
		}

		return implode(' ', $arr);
	}

	/**
	 * @return array|DaoNavigationItem[]
	 */
	public function getPath(): array
	{
		$result = [$this];

		$parent = $this->parent;
		while ($parent) {
			$result[] = $parent;

			$parent = $parent->parent;
		}

		return array_reverse($result);
	}

	public function getLink(): string
	{
		if (is_array($this->link)) {
			return $this->link['link'] ?? '';
		}

		return $this->link ?: '';
	}
}
