<?php declare(strict_types = 1);

namespace Core\Console\Content;

use Core\AdminModule\Model\EditorDataService;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Parameters;
use Core\Model\Sites;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class CheckMissingEditorData extends AbstractCommand
{
	protected EditorDataService  $editorDataService;
	protected MailBuilderFactory $mailBuilderFactory;
	protected Sites              $sites;

	protected array $aclData = [];

	public function __construct(
		EditorDataService  $editorDataService,
		MailBuilderFactory $mailBuilderFactory,
		Sites              $sites
	)
	{
		$this->editorDataService  = $editorDataService;
		$this->mailBuilderFactory = $mailBuilderFactory;
		$this->sites              = $sites;

		parent::__construct('core:content:check-missing-editor-data');
	}

	protected function configure(): void
	{
		$this->setName('core:content:check-missing-editor-data')->setDescription('Check missing editor data. (Filter editorData)');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui)
	{
		$output->writeln('Checking missing editor data...');

		try {
			$dataForSave = $this->editorDataService->runCheck();
			$emails      = Parameters::load('editorData.sendNotificationToIfMissing') ?: [];

			if (!empty($dataForSave) && !empty($emails)) {
				foreach ($this->sites->getSites() as $site) {
					if (!$site->isVisible || !$site->getEmail()) {
						continue;
					}

					$mail = $this->mailBuilderFactory->create($site->getIdent());
					$mail->setSubject('aa');
					$mail->setTemplateFile(__DIR__ . '/../../AdminModule/Templates/_emails/missingEditorData.latte');
					$mail->setParameters(['data' => $dataForSave]);

					$mail->setFrom($site->getEmail(), $site->getSiteName() ?: '');
					foreach ($emails as $email) {
						$mail->addTo($email);
					}
					$mail->send();

					break;
				}
			}

			$output->writeLn('<info>[OK] - Done</info>');
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 0;
	}
}
