<?php declare(strict_types = 1);

namespace Core\Console\Users;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Lang\Langs;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Users\Model\Entities\Privilege;
use Users\Model\Entities\Resource;

class UpdateAcl extends AbstractCommand
{
	protected EntityManagerDecorator $em;
	protected Langs                  $langs;

	protected array $aclData = [];

	public function __construct(
		EntityManagerDecorator $em,
		Langs                  $langs
	)
	{
		$this->em    = $em;
		$this->langs = $langs;

		parent::__construct('users:update-acl');
	}

	public function setAclData(array $data): void
	{
		$this->aclData = $data;
	}

	protected function configure(): void
	{
		$this->setName('users:update-acl')->setDescription('Update acl data');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui)
	{
		$output->writeln('UPDATE ACL');

		try {
			$resources       = $this->em->getRepository(Resource::class)->findPairs([], 'name', [], 'id');
			$configResources = array_keys($this->aclData);

			foreach ($this->langs->getLangs(false) as $k => $lang) {
				$configResources[] = 'locale:' . $k;
			}

			$deleteResources = (array_diff($resources, $configResources));
			$addResources    = (array_diff($configResources, $resources));

			$privileges       = $this->em->getRepository(Privilege::class)->findPairs([], 'name', [], 'id');
			$configPrivileges = [];

			foreach ($this->aclData as $arr) {
				foreach ($arr as $v) {
					$configPrivileges[$v] = $v;
				}
			}

			$configPrivileges['deepLTranslate'] = 'deepLTranslate';

			$deletePrivileges = (array_diff($privileges, $configPrivileges));
			$addPrivileges    = (array_diff($configPrivileges, $privileges));

			if ($deletePrivileges || $deleteResources || $addPrivileges || $addResources) {
				$this->em->beginTransaction();
				try {
					if ($deleteResources) {
						$this->em->createQuery("DELETE " . Resource::class . " r WHERE r.name IN (:names)")->setParameter('names', $deleteResources)->execute();
					}

					if ($deletePrivileges) {
						$this->em->createQuery("DELETE " . Privilege::class . " p WHERE p.name IN (:names)")->setParameter('names', $deletePrivileges)->execute();
					}

					if ($addResources) {
						foreach ($addResources as $resource) {
							$this->em->persist(new Resource($resource));
						}
					}

					if ($addPrivileges) {
						foreach ($addPrivileges as $privilege) {
							$this->em->persist(new Privilege($privilege));
						}
					}

					$this->em->flush();
					$this->em->commit();
					$output->writeLn('<info>[OK] - data updated</info>');
				} catch (\Exception $e) {
					$this->em->rollback();
					$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
				}
			} else {
				$output->writeLn('<info>[OK] - nothing to update</info>');
			}
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 0;
	}
}
