<?php declare(strict_types = 1);

namespace Core\Console\Utils;

use Core\Model\Parameters;
use Core\Model\SystemConfig;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Nette\Localization\Translator;
use Nette\Mail\Mailer;
use Nette\Mail\Message;
use Nette\Utils\FileSystem;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Tracy\Debugger;

class CoreModifyLastSystemUpdate extends AbstractCommand
{
	protected Mailer     $mailer;
	protected Translator $translator;

	public function __construct(
		Mailer     $mailer,
		Translator $translator
	)
	{
		parent::__construct();
		$this->mailer     = $mailer;
		$this->translator = $translator;
	}

	protected function configure(): void
	{
		$this->setName('core:modify-last-system-update')
			->addOption('emailNotify', null, InputOption::VALUE_OPTIONAL, 'Send notify to defined emails', false);
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui)
	{
		$output->writeln('<comment>--- Modify last system update ---</comment>');
		$emailNotify = (bool) $input->getOption('emailNotify');

		try {
			FileSystem::write(DATA_DIR . DS . 'core' . DS . 'lastSystemUpdate', (string) time());
			$output->writeLn('<info>[OK] - modified</info>');

			if ($emailNotify) {
				$subject = $this->translator->translate('core.systemUpdate.notifyEmailSubject', ['siteName' => Parameters::load('siteName')]);
				$text    = $this->translator->translate('core.systemUpdate.notifyEmailText');

				foreach (SystemConfig::load('systemUpdateNotifyEmails') as $email) {
					try {
						$message = new Message;
						$message->setFrom('noreply@pshk.cz', 'PSHK CMS');
						$message->addTo($email);
						$message->setSubject($subject);
						$message->setHtmlBody($text);

						$this->mailer->send($message);
						$output->writeLn('<info>[OK] - notify sent - ' . $email . '</info>');
					} catch (\Exception $e) {
						$output->writeln('<error>[ERROR] - notify failed sent to ' . $email . ' - ' . $e->getMessage() . '</error>');
						Debugger::log($e, '_coreSystemUpdateNotify');
					}
				}
			}
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
			Debugger::log($e, '_coreSystemUpdate');
		}

		return 0;
	}
}
