<?php declare(strict_types = 1);

namespace Core\DI;

use Core\FrontModule\Model\ABTest;
use Core\FrontModule\Model\Providers\ISearchItem;
use Core\Model\Images\Macros\Macros;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\Translation\Loaders\NeonDoctrineMixedLoader;
use Nette;
use Nette\PhpGenerator\ClassType;
use Nette\Utils\FileSystem;
use Nettrine\ORM\DI\Helpers\MappingHelper;
use Tracy\Debugger;

class CoreExtension extends CompilerExtension
{
	public const TAG_ROUTER = 'app.router';

	public function loadConfiguration(): void
	{
		parent::loadConfiguration();
		$this->setConfig($this->loadFromFile(__DIR__ . '/config.neon'));;

		$builder = $this->getContainerBuilder();
		$builder->addDefinition('application.presenterFactoryCallback')
			->setFactory(
				'Nette\Bridges\ApplicationDI\PresenterFactoryCallback',
				[
					'@container',
					Nette\Application\UI\Presenter::INVALID_LINK_WARNING,
					null,
				]
			);


		$builder->removeDefinition('application.presenterFactory');
		$builder->addDefinition('application.presenterFactory')
			->setFactory(
				'\Core\Model\Application\PresenterFactory',
				['@application.presenterFactoryCallback']
			);

		$builder->addDefinition('translation.neonLoader')
			->setFactory(NeonDoctrineMixedLoader::class);
	}

	public function beforeCompile(): void
	{
		FileSystem::createDir(LOG_DIR . '/scheduler');
		FileSystem::createDir(ROOT_DIR . '/migrations');
		FileSystem::createDir(ABTest::$defaultDir);

		parent::beforeCompile();

		$this->addRouters();
		$this->setMapping(['Core' => 'Core\*Module\Presenters\*Presenter']);

		if (isset($this->compiler->getConfig()['application']->mapping)) {
			$this->setMapping($this->compiler->getConfig()['application']->mapping);
		}

		if (!defined('DATA_DIR')) {
			throw new \Exception('Constant DATA_DIR missing in Bootstrap file. Default is "ROOT_DIR . \'/data\'"');
		} else {
			FileSystem::createDir(DATA_DIR);
		}

		$builder = $this->getContainerBuilder();

		$builder->getDefinition('nette.latteFactory')
			->addSetup(Macros::class . '::install(?->getCompiler(), ?)', [
				'@self',
				$builder->parameters['system']['images'],
			]);

		$builder->getDefinition('nette.latteFactory')
			->addSetup('addProvider', ['templateText', $builder->getDefinition('provider.templateText')])
			->addSetup('addProvider', ['imagePipe', $builder->getDefinition('imagePipe')])
			->addSetup('addProvider', ['linkGenerator', $builder->getDefinition('application.linkGenerator')])
			->addSetup(new Nette\DI\Definitions\Statement('$service->addProvider(\'lang\', ?->getLocale())', [$builder->getDefinition('translation.translator')]))
			->addSetup(new Nette\DI\Definitions\Statement('$service->addProvider(\'site\', ?->getCurrentSite())', [$builder->getDefinition('core.sites')]))
			->addSetup(new Nette\DI\Definitions\Statement('$service->addProvider(\'siteSettings\', ?->readRawData(
				?->getPrefixed(?->getCurrentSite()->getIdent() . \'_webSettings\') \?: [],
				Core\Model\SystemConfig::load(?->getCurrentSite()->getIdent() . \'_webSettings\')
			))', [
				$builder->getDefinition('templateReader'),
				$builder->getDefinition('core.settings'),
				$builder->getDefinition('core.sites'),
				$builder->getDefinition('core.sites'),
			]))
			->addSetup(new Nette\DI\Definitions\Statement('$service->addProvider(\'getTemplateTextValues\', function($v) {return ?->getTemplateTextValues($v);})', [
				$builder->getDefinition('templateReader'),
			]))
			->addSetup(new Nette\DI\Definitions\Statement('$service->addProvider(\'webSettings\', ?->readRawData(
				?->getAll() \?: [],
				Core\Model\SystemConfig::load(\'globalData\')
			))', [
				$builder->getDefinition('templateReader'),
				$builder->getDefinition('core.settings'),
			]))
			->addSetup(new Nette\DI\Definitions\Statement('$service->addProvider(\'webColors\', ?->getColors(?->getCurrentSite()->getIdent()))', [
				$builder->getDefinition('core.webColors'),
				$builder->getDefinition('core.sites'),
			]));

		$builder->getDefinition('templateTextTypesCollection')
			->addSetup(new Nette\DI\Definitions\Statement('$service->setItems(?)', [$builder->findByType(ITemplateTextType::class)]));

		$builder->getDefinition('core.front.searchItemsCollection')
			->addSetup(new Nette\DI\Definitions\Statement('$service->loadItems(?)', [$builder->findByType(ISearchItem::class)]));

		$builder->getDefinition('usersUpdateAcl')
			->addSetup('setAclData', [$builder->parameters['acl'] ?: []]);

		$builder->getDefinition('core.admin.components.navigation')
			->addSetup('setData', [$builder->parameters['adminNavigation']]);

		$builder->getDefinition('translation.translator')
			->addSetup('addLoader', ['neon', $builder->getDefinition('translation.neonLoader')]);

		$builder->getDefinition('langs')
			->addSetup('setTranslator', [$builder->getDefinition('translation.translator')]);

		foreach (glob(ROOT_DIR . '/custom/*', GLOB_ONLYDIR) ?: [] as $dir) {
			$entitiesDir = $dir . '/Model/Entities';
			if (file_exists($entitiesDir)) {
				MappingHelper::of($this)
					->addAnnotation(basename($dir) . '\Model\Entities', $entitiesDir);
			}
		}
	}

	/**
	 * @param ClassType $class
	 */
	public function afterCompile(ClassType $class)
	{
		parent::afterCompile($class);

		$init = $class->methods["initialize"];

		$body = 'if (isset($_SERVER[\'HTTP_HOST\']) && strpos((string) $_SERVER[\'HTTP_HOST\'], \'admin\') === 0) Core\Model\Module::setIsAdmin();' . "\n";
		$body .= 'elseif (strpos((string) $_SERVER[\'REQUEST_URI\'], \'/admin\') === 0) Core\Model\Module::setIsAdmin();' . "\n";
		$body .= 'elseif (strpos((string) $_SERVER[\'SERVER_NAME\'], \'admin.\') === 0) Core\Model\Module::setIsAdmin();' . "\n";
		$body .= 'Core\Model\Helpers\DatabaseConfig::$config = $this->parameters[\'database\'];' . "\n";
		$body .= 'if (php_sapi_name() === \'cli\' && isset($_SERVER[\'argv\']) && is_array($_SERVER[\'argv\'])) {
				foreach ($_SERVER[\'argv\'] as $v) {
					if (\Core\Model\Helpers\Strings::startsWith((string) $v, \'--site=\')) {
						\Core\Model\Sites::$currentIdentOverride = substr((string) $v, 7);
					} else if (\Core\Model\Helpers\Strings::startsWith((string) $v, \'--lang=\')) {
						\Core\Model\Sites::$currentLangOverride = substr((string) $v, 7);
						$lang = substr((string) $v, 7);
						$this->getService(\'langs\')->setDefault($lang);
						$this->getService(\'translation.translator\')->setLocale($lang);
					}
				}
			}' . "\n";
		$body .= '$this->getService(\'langs\')->setTranslator($this->getService(\'translation.translator\'));' . "\n";
		$body .= 'Core\Model\Parameters::setParams($this->parameters);' . "\n";
		$body .= 'Core\Model\SystemConfig::setParams($this->parameters[\'system\']);' . "\n";
		$body .= '$this->getService(\'core.sites\')->validateAndRedirectSite();' . "\n";

		if (Debugger::$productionMode) {
			$body .= 'DEFINE("BUILD_TIME", "' . time() . '");';
		} else {
			$body .= 'DEFINE("BUILD_TIME", time());';
		}

		$body .= $init->getBody();
		$init->setBody($body);

		$builder = $this->getContainerBuilder();

		$init->addBody('\Core\Model\UI\Form\Controls\FilesManagerInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('\Core\Model\UI\Form\Controls\EditorInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('\Core\Model\UI\Form\Controls\LangsSelectInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('\Core\Model\UI\Form\Controls\GalleryPopupInput::register($this->getService(?));', [
			$this->getContainerBuilder()->getByType('\Nette\DI\Container'),
		]);

		$init->addBody('define("CORE_TEST_MODE", $this->parameters["system"]["testMode"]);');
		$init->addBody('$this->getService(\'defaultLang\')->locale = $this->getService(\'translation.translator\')->getLocale();');

		try {
			$navigationFactory = $class->getMethod('createServiceCore__navigationFactory');
			$navigationFactory->setBody(str_replace('return $service;', '$service->setData($this->container->parameters["navigation"]);' . "\n" . 'return $service;', $navigationFactory->getBody()));
		} catch (\Exception $e) {
		}

		$init->addBody('
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$getUserEmails = function() { return $this->getService(\'users.users\')->getUserMailWithLogPrivilege(); };
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$site = php_sapi_name() !== \'cli\' \? $this->getService(\'core.sites\')->getCurrentSite() : null;
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$translator = $this->getService(\'translation.translator\');
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$mailer = $this->getService(\'mail.mailer\');
			Core\Model\Notifiers\MailNotifiers\LogNotifier::$developersMail = ?;
		', [
			$builder->parameters['system']['error']['email'],
		]);

		$init->addBody('Core\Model\Images\ImagePreviewFactory::$imagePipe = $this->getService(\'imagePipe\');');

		$init->addBody('$this->getService(\'appState\')->init();');
		$init->addBody('Core\Model\Helpers\CoreHelper::setCompiled();');
	}

	private function addRouters(): void
	{
		$builder = $this->getContainerBuilder();
		// Get application router
		$router = $builder->getDefinition('router');
		// Init collections
		$routerFactories = [];
		foreach ($builder->findByTag(self::TAG_ROUTER) as $serviceName => $priority) {
			// Priority is not defined...
			if (is_bool($priority)) {
				// ...use default value
				$priority = 100;
			}
			$routerFactories[$priority][$serviceName] = $serviceName;
		}

		// Sort routes by priority
		if (!empty($routerFactories)) {
			krsort($routerFactories, SORT_NUMERIC);
			$routerFactories = array_reverse($routerFactories, true);

			foreach ($routerFactories as $priority => $items) {
				$routerFactories[$priority] = $items;
			}

			// Process all routes services by priority...
			foreach ($routerFactories as $priority => $items) {
				// ...and by service name...
				foreach ($items as $serviceName) {
					$factory = new Nette\DI\Definitions\Statement(['@' . $serviceName, 'createRouter']);
					$router->addSetup('offsetSet', [null, $factory]);
				}
			}
		}
	}

	public function getTranslationResources(): array
	{
		$resources   = parent::getTranslationResources();
		$resources[] = APP_DIR . '/lang';

		return $resources;
	}
}
