<?php declare(strict_types = 1);

namespace Core\Model\Images;

use Core\Model\Helpers\Strings;
use Core\Model\SystemConfig;
use Nette\Utils\FileSystem;
use Nette\Utils\Image as NImage;
use Nette\Utils\Json;
use Nette\Utils\UnknownImageFileException;
use Tracy\Debugger;

class ImageHelper
{
	/**
	 * @param string $file
	 * @param mixed  $width
	 * @param mixed  $height
	 * @param int    $flags
	 *
	 * @return string
	 * @throws UnknownImageFileException
	 */
	public static function createBase64Miniature(string $file, $width, $height, $flags = NImage::FIT): string
	{
		$image = NImage::fromFile($file);
		$image->resize($width, $height, $flags);

		$type   = pathinfo($file, PATHINFO_EXTENSION);
		$base64 = base64_encode((string) $image);

		return sprintf('data:image/%s;base64,%s', $type, $base64);
	}

	public static function supportWebP(): bool
	{
		return true;
	}

	public static function isImage(string $file): bool
	{
		return \Nette\Utils\Image::detectTypeFromFile($file) !== null;
	}

	public static function autoResize(string $file, int $width = null, int $height = null): void
	{
		if (!SystemConfig::load('images.autoResize.allow') || !self::isImage($file)) {
			return;
		}

		$width  = (int) ($width ?: SystemConfig::load('images.autoResize.w'));
		$height = (int) ($height ?: SystemConfig::load('images.autoResize.h'));

		try {
			$image   = NImage::fromFile($file);
			$oWidth  = $image->getWidth();
			$oHeight = $image->getHeight();

			if ($oWidth <= $width && $oHeight <= $height) {
				return;
			}

			$image->resize($width, $height, NImage::SHRINK_ONLY);
			$image->save($file);
			Debugger::log("resize $file from {$oWidth}x{$oHeight} to {$image->getWidth()}x{$image->getHeight()}", '_imagesAutoResize');
		} catch (\Exception $e) {
			Debugger::log("resize failed $file", '_imagesAutoResizeFailed');
			$imageSize = @getimagesize($file);
			Debugger::log("imageSize - " . Json::encode($imageSize), '_imagesAutoResizeFailed');

			try {
				if (isset($imageSize['mime']) && $imageSize['mime'] === 'image/tiff') {
					FileSystem::delete($file);
				}
			} catch (\Exception $e) {
			}
		}
	}
}
