<?php declare(strict_types = 1);

namespace Core\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\SiteText;
use Core\Model\Helpers\Strings;
use Core\Model\Router\RouteHelper;
use Doctrine\DBAL\Exception;
use Navigations\Model\NavigationConfig;
use Nette\Application\BadRequestException;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\Request;
use Nette\Http\Response;
use Nette\Http\Url;

class Sites
{
	public const CACHE_NAMESPACE = 'sites';

	protected EntityManagerDecorator $em;
	protected Request                $httpRequest;
	protected Response               $httpResponse;
	protected Translator             $translator;
	protected Cache                  $cache;

	/** @var Dao\Site[][] */
	protected $cSites;

	protected ?Dao\Site $cCurrent = null;

	public static ?string    $currentIdentOverride = null;
	public static ?string    $currentLangOverride  = null;
	protected static ?string $siteValidatedLang    = null;

	protected static ?Dao\Site $cliSite = null;

	public function __construct(
		EntityManagerDecorator $em,
		Request                $request,
		Response               $response,
		Translator             $translator,
		Storage                $cacheStorage
	)
	{
		$this->em           = $em;
		$this->httpRequest  = $request;
		$this->httpResponse = $response;
		$this->translator   = $translator;
		$this->cache        = new Cache($cacheStorage, self::CACHE_NAMESPACE);
	}

	public function getCurrentSite(): Dao\Site
	{
		if (php_sapi_name() === 'cli' && !self::$currentIdentOverride && isset($_SERVER['argv']) && is_array($_SERVER['argv'])) {
			foreach ($_SERVER['argv'] as $v) {
				if (Strings::startsWith((string) $v, '--site=')) {
					self::$currentIdentOverride = substr((string) $v, 7);
				} else if (Strings::startsWith((string) $v, '--lang=')) {
					self::$currentLangOverride = substr((string) $v, 7);
					$lang                      = substr((string) $v, 7);
					$this->translator->setLocale($lang);
				}
			}
		}

		if (php_sapi_name() === 'cli' && !self::$currentIdentOverride) {
			$ident  = $this->em->getConnection()->fetchOne("SELECT ident FROM core__site LIMIT 1");
			$site   = new Dao\Site($ident);
			$domain = new Dao\SiteDomain($site, '//php', 'cs', 1);

			self::$cliSite = $site;

			return $site;
		}

		try {
			$current = $this->cCurrent;

			if (self::$currentIdentOverride) {
				$current = $this->getSites()[self::$currentIdentOverride];
			}

			if (!$current) {
				$host          = $this->httpRequest->getUrl()->getHost();
				$foundSite     = null;
				$alternateSite = null;

				foreach ($this->getSites() as $site) {
					$domain = $site->getDomains()[$this->translator->getLocale()] ?? null;

					if ($domain && $host == $domain->getDomain()) {
						$site->currentLang = $domain->getLang();
						$foundSite         = $site;
						break;
					}

					if (!$alternateSite) {
						foreach ($site->getDomains() as $domain) {
							if ($host == $domain->getDomain()) {
								$site->currentLang = $domain->getLang();
								$alternateSite     = $site;
							}
						}
					}
				}

				if (!$alternateSite) {
					$alternateSite = array_values($this->getSites())[0];
				}

				$this->cCurrent = $foundSite ?: $alternateSite;
			} else {
				$this->cCurrent = $current;
			}

			if (isset($this->cCurrent->getDomains()[self::$siteValidatedLang])) {
				$this->cCurrent->currentLang = self::$siteValidatedLang;
			}

			if (self::$currentLangOverride && isset($this->cCurrent->getDomains()[self::$currentLangOverride])) {
				$this->cCurrent->currentLang = self::$currentLangOverride;
			}

			return $this->cCurrent;
		} catch (Exception $e) {
		}

		throw new \Exception("Site not found");
	}

	public function validateAndRedirectSite(): void
	{
		if (php_sapi_name() === 'cli') {
			return;
		}

		foreach (Parameters::load('system.skipValidateAndRedirectRequestUriStart') ?: [] as $skip) {
			if (Strings::startsWith((string) $_SERVER['REQUEST_URI'], $skip)) {
				return;
			}
		}

		$host      = $this->httpRequest->getUrl()->getHost();
		$hostParts = explode('.', $host, 2);
		$pathParts = explode('/', ltrim($this->httpRequest->getUrl()->getPath(), '/'));
		$urlLang   = strlen($pathParts[0]) === 2 ? $pathParts[0] : null;

		if (NavigationConfig::load('useCZinUrl') && $urlLang === 'cz') {
			$urlLang = 'cs';
		}

		foreach ($this->getSites() as &$site) {
			foreach ($site->getDomains() as $lang => $domain) {
				$hostFound = false;

				if ($host === $domain->domain) {
					$hostFound = 'prod';
				} else if ($host === 'admin.' . str_replace('www.', '', $domain->domain)) {
					$hostFound = 'prod';
				}

				if ($hostFound !== false) {
					if ($urlLang && $urlLang == $domain->getLang() && $domain->isDefault && !NavigationConfig::load('showDefaultLangInUrl')) {
						header('Location: https://' . $domain->domain);
						exit;
					} else if (!$urlLang) {
						self::$siteValidatedLang = $domain->getLang();
						$this->setNavigationLang($domain->getLang());
						$site->currentLang = $domain->getLang();
						$this->translator->setLocale($domain->getLang());
					}

					if ($urlLang) {
						if ($urlLang != $domain->getLang()) {
							$tmp       = $site->getDomains()[$urlLang] ?? null;
							$tmpDomain = $tmp ? $tmp->domain : null;
							if ($tmp && $tmpDomain && $tmpDomain !== $host) {
								header('Location: https://' . $tmpDomain);
								exit;
							}

							if ($tmp) {
								$urlLang = $tmp->getLang();
							} else {
								$urlLang = $domain->getLang();
							}

							self::$siteValidatedLang = $urlLang;
							$this->setNavigationLang($urlLang);
							$site->currentLang = $urlLang;
							$this->translator->setLocale($urlLang);
						}
					}

					return;
				}

				$parts = explode('.', $domain->getDomain(), 2);

				// host je bez www ale doména s
				if (isset($parts[1]) && $host === $parts[1]) {
					$this->redirectToHost($domain->domain);
				}

				// host je s www ale doména bez
				if ($hostParts[1] === $domain->getDomain()) {
					$this->redirectToHost($domain->domain);
				}
			}
		}

		throw new BadRequestException('domainNotFound');
	}

	protected function setNavigationLang(string $lang): void
	{
		if ($lang === 'cz') {
			$lang = 'cs';
		}

		RouteHelper::$validatedLang = $lang;
	}

	/**
	 * @return Dao\Site[]
	 */
	public function getSites(bool $onlyActive = true): array
	{
		$cKey = $onlyActive ? 'active' : 'all';
		/** @phpstan-ignore-next-line */
		if (!array_key_exists($cKey, $this->cSites ?? []) || $this->cSites[$cKey] === null) {
			$this->cSites[$cKey] = [];

			/** @var Dao\Site[] $data */
			$data = $this->cache->load(self::CACHE_NAMESPACE, function(&$dep) {
				$dep = [
					Cache::EXPIRATION => '1 month',
				];

				$data = [];

				$qb = $this->em->getRepository(SiteText::class)->createQueryBuilder('st')
					->addSelect('s.ident, st.domain, st.siteName, s.siteNameSeparator, st.logo, st.inverseLogo, s.email, st.lang, 
						st.isDefault, st.isActive, st.defaultCountry, st.defaultCurrency, st.disableIndex, s.isVisible')
					->innerJoin('st.site', 's')
					->orderBy('st.isDefault', 'DESC');

				foreach ($qb->getQuery()->getScalarResult() as $row) {
					if (!isset($data[$row['ident']])) {
						$site            = new Dao\Site($row['ident']);
						$site->email     = $row['email'];
						$site->isVisible = (bool) $row['isVisible'];

						$data[$row['ident']] = $site;
					} else {
						$site = $data[$row['ident']];
					}

					$domain                    = new Dao\SiteDomain($site, $row['domain'], $row['lang'], (int) $row['isDefault']);
					$domain->siteName          = $row['siteName'];
					$domain->logo              = $row['logo'] ?: null;
					$domain->inverseLogo       = $row['inverseLogo'] ?: null;
					$domain->siteNameSeparator = $row['siteNameSeparator'];
					$domain->isActive          = (bool) $row['isActive'];
					$domain->defaultCountry    = $row['defaultCountry'] ?? null;
					$domain->defaultCurrency   = $row['defaultCurrency'] ?? null;
					$domain->disableIndex      = (bool) $row['disableIndex'];
				}

				return $data;
			});

			if ($onlyActive) {
				$arr = [];

				foreach ($data as $k => &$site) {
					foreach ($site->getDomains() as $domain) {
						if (!$domain->isActive) {
							$site->removeDomain($domain->getLang());
						}
					}

					if ($site->getDomains()) {
						$arr[$k] = $site;
					}
				}

				$this->cSites[$cKey] = $arr;
			} else {
				$this->cSites[$cKey] = $data;
			}
		}

		return $this->cSites[$cKey];
	}

	protected function redirectToHost(string $host, ?string $path = null): void
	{
		$url    = $this->httpRequest->getUrl();
		$newUrl = new Url($url);
		$newUrl->setHost($host);
		$newUrl->setPath($path ?: '');

		$this->httpResponse->redirect($newUrl->getAbsoluteUrl());
		exit;
	}
}
