<?php declare(strict_types = 1);

namespace Core\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\StaticTextTranslation;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Translation\Utils\Helpers;
use Exception;
use Nette\DI\Container;
use Nette\Neon\Neon;
use Nette\Utils\FileSystem;

/**
 * @method StaticTextTranslation[] getAll()
 */
class StaticTexts extends BaseEntityService
{
	/** @var string */
	protected $entityClass = StaticTextTranslation::class;

	/** @var Translator */
	protected $translator;

	/** @var Container */
	protected $container;

	/** @var array */
	protected $resources;

	public function __construct(
		Translator $translator,
		Container  $container
	)
	{
		$this->translator = $translator;
		$this->container  = $container;
	}

	/**
	 * @param string $locale
	 * @param string $path
	 */
	public function addResource(string $locale, string $domain, string $path): void
	{
		$this->resources[$locale][$path] = $domain;
	}

	/**
	 * @param string $domain
	 * @param string $locale
	 *
	 * @return array
	 */
	public function getByDomainLocale(string $domain, string $locale): array
	{
		$qb = $this->getEr()->createQueryBuilder('st');
		$qb->select('st.id, st.message')->where('st.id LIKE :key AND st.locale = :locale');
		$qb->setParameters([
			'key'    => $domain . '%',
			'locale' => $locale,
		]);

		return $qb->getQuery()->getResult();
	}

	/**
	 * @param string $key
	 * @param string $locale
	 *
	 * @return StaticTextTranslation|null
	 */
	public function getOneByKeyLocale(string $key, string $locale): ?StaticTextTranslation
	{
		/** @var StaticTextTranslation|null $entity */
		$entity = $this->getEr()->findOneBy(['id' => $key, 'locale' => $locale]);

		return $entity;
	}

	/**
	 * @return array
	 */
	protected function getResources(): array
	{
		$result = [];

		foreach ($this->resources as $locale => $res) {
			foreach ($res as $resource => $domain) {
				if (!Helpers::isDomainAllowed($domain, $this->container->getParameters()['system']['staticTextsTranslation']['domains'])) {
					continue;
				}
				$result[$domain][$locale] = $resource;
			}
		}

		return $result;
	}

	/**
	 * @return array
	 */
	public function getMergedMessages(): array
	{
		return $this->getMessages(function(string $domain, string $locale, array $content) {
			foreach ($this->getByDomainLocale($domain, $locale) as $value) {
				if (isset($content[$value['id']])) {
					$content[$value['id']] = $value['message'];
				}
			}

			return $content;
		});
	}

	/**
	 * @return array
	 */
	public function getMessagesWithResources(): array
	{
		return $this->getMessages(function(string $domain, string $locale, array $content) {

			$tmpArr = [];
			foreach ($content as $k => $v) {
				$tmpArr[$k] = ['file' => $v, 'db' => null];
			}

			foreach ($this->getByDomainLocale($domain, $locale) as $value) {
				if (isset($tmpArr[$value['id']])) {
					$tmpArr[$value['id']]['db'] = $value['message'];
				}
			}

			return $tmpArr;
		});
	}

	/**
	 * @return array
	 */
	public function getFileMessages(): array
	{
		return $this->getMessages();
	}

	/**
	 * @return array
	 */
	public function getDbMessages(): array
	{
		$delimiter  = '.';
		$dbResource = [];
		foreach ($this->getAll() as $item) {
			$dbResource[explode($delimiter, $item->id)[0]][$item->locale][$item->id] = $item->message;
		}

		return $dbResource;
	}

	/**
	 * @return array
	 */
	public function merge(): array
	{
		$delimiter = '.';

		// loaded neon
		$fileResource = $this->getFileMessages();

		// loaded db
		$dbResource = $this->getDbMessages();

		$flattenDbResource   = Arrays::flattenWithDottedKeys($dbResource);
		$flattenFileResource = Arrays::flattenWithDottedKeys($fileResource);

		// was deleted from file, remove from db
		foreach (array_diff_key($flattenDbResource, $flattenFileResource) as $key => $val) {
			[$domain, $locale, $id] = explode($delimiter, $key, 3);
			$entity = $this->getOneByKeyLocale($id, $locale);

			if ($entity) {
				$this->em->remove($entity);
			}
		}

		$this->em->flush();

		return $dbResource ?: [];
	}

	/**
	 * @param callable|null $contentDecorator
	 *
	 * @return array
	 */
	private function getMessages(callable $contentDecorator = null): array
	{
		$result = [];
		foreach ($this->getResources() as $domain => $resources) {
			foreach ($resources as $locale => $resource) {
				$fileContent = @file_get_contents($resource);

				if (!$fileContent) {
					continue;
				}

				$fileDecoded = Neon::decode($fileContent);

				if ($fileDecoded) {
					$array = Arrays::flattenWithDottedKeys($fileDecoded, $domain . '.');

					if ($contentDecorator) {
						$array = $contentDecorator($domain, $locale, $array);
					}

					$result[$domain][$locale] = $array;
				}
			}

			if (isset($result[$domain]['cs_CZ'])) {
				$result[$domain] = ['cs_CZ' => $result[$domain]['cs_CZ']] + $result[$domain];
			}
		}

		return $result;
	}

	public function clearCache(): void
	{
		$cacheDir = $this->translator->getCacheDir();
		if ($cacheDir !== null && file_exists($cacheDir)) {
			try {
				FileSystem::delete($cacheDir);
			} catch (Exception $e) {
			}
		}
	}

}
