<?php declare(strict_types = 1);

namespace Core\Model\Subscribers;

use Core\Model\Event\TemplateFactoryEvent;
use Core\Model\Event\TTPreload;
use Core\Model\SystemConfig;
use Core\Model\TemplateReader\TemplateReader;
use DynamicModule\FrontModule\Model\Dao\Group;
use DynamicModule\FrontModule\Model\Dao\Member;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class TemplateSubscriber implements EventSubscriberInterface
{
	protected TemplateReader $templateReader;

	protected array $webSettings  = [];
	protected array $siteSettings = [];

	protected static bool $isProcessing = false;

	public function __construct(TemplateReader $templateReader)
	{
		$this->templateReader = $templateReader;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'templateFactory::createTemplate' => ['createTemplate', 250],
			'core.ttPreload'                  => 'ttPreload',
		];
	}

	public function createTemplate(TemplateFactoryEvent $event): void
	{
		$siteIdent = $event->template->currentSite
			? $event->template->currentSite->getIdent()
			: '';

		if ($event->template->webSettings) {
			if (!array_key_exists($siteIdent, $this->webSettings)) {
				$this->webSettings[$siteIdent] = $this->templateReader->readRawData($event->template->webSettings ?: [], SystemConfig::load('globalData'));
			}

			$event->template->webSettings = $this->webSettings[$siteIdent];
		}

		if ($event->template->siteSettings && $siteIdent) {
			if (!array_key_exists($siteIdent, $this->siteSettings)) {
				$this->siteSettings[$siteIdent] = $this->templateReader->readRawData(
					$event->template->siteSettings ?: [],
					SystemConfig::load($siteIdent . '_webSettings')
				);
			}

			$event->template->siteSettings = $this->siteSettings[$siteIdent];
		}
	}

	public function ttPreload(TTPreload $event): void
	{
		if (!SystemConfig::load('allowProcessTTPreload')) {
			return;
		}

		if (self::$isProcessing) {
			return;
		}
		self::$isProcessing = true;

		// Projde polozky dynamnickeho modulu a nastavit do hodnot spravne hodnoty z tt
		$loopDynModule = function(array $rows) use (&$loopDynModule) {
			foreach ($rows as $row) {
				$templateInputs = [];
				$data           = [];

				if ($row instanceof Group && $row->module) {
					$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateGroup/', TEMPLATES_DIR, ucfirst($row->module)));
					$templateInputs = $this->templateReader->getTemplateInputs($row->getTemplate());

					foreach ($row->getFields() as $field) {
						if (is_object($field)) {
							$data[$field->getKey()] = $field->getValue();
						}
					}
				} else if ($row instanceof Member && $row->module) {
					$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateMember/', TEMPLATES_DIR, ucfirst($row->module)));
					$templateInputs = $this->templateReader->getTemplateInputs($row->getTemplate());

					foreach ($row->getFields() as $field) {
						if (is_object($field)) {
							$data[$field->getKey()] = $field->getValue();
						}
					}
				}

				if (!empty($templateInputs) && !empty($data)) {
					$data = $this->templateReader->readRawData($data, $templateInputs);

					if ($row instanceof Member || $row instanceof Group) {
						foreach ($row->getFields() as &$field) {
							if (is_object($field) && isset($data[$field->getKey()])) {
								$field->setValue($data[$field->getKey()]);
							}
						}
					}
				}

				if ($row instanceof Group) {
					if ($row->getChildren()) {
						$loopDynModule($row->getChildren());
					}

					if ($row->getMembers()) {
						$loopDynModule($row->getMembers());
					}
				}
			}
		};

		if ($event->rawData) {
			$loopDynModule($event->rawData);
		}

		self::$isProcessing = false;
	}
}
