<?php declare(strict_types = 1);

namespace Core\Model\Templating\Macros;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\Module;
use Core\Model\Parameters;
use Core\Model\SystemConfig;
use Latte\Compiler;
use Latte\MacroNode;
use Latte\Macros\MacroSet;
use Latte\PhpWriter;
use Nette\Utils\Json;

class TemplateText extends MacroSet
{
	public static function install(Compiler $compiler): void
	{
		$self = new self($compiler);
		$self->addMacro('templateText', [$self, 'macroTemplateText']);
		$self->addMacro('tt', [$self, 'macroTemplateText']);
		$self->addMacro('templateTextR', [$self, 'macroTemplateTextR']);
		$self->addMacro('ttR', [$self, 'macroTemplateTextR']);
	}

	public function macroTemplateText(MacroNode $node, PhpWriter $writer): string
	{
		if (SystemConfig::load('useLatteTemplateReader') && Module::isAdmin()) {
			return $this->macroTemplateTextR($node, $writer);
		}

		$oData = self::parseInputAttrs($node->args);

		if (!isset($oData['name'])) {
			return '';
		}

		$str  = '';
		$data = $this->prepareData($oData, "'");

		if (Module::isFront()) {
			unset($data['items']);
		}

		if (SystemConfig::load('useLatteTemplateReader')) {
			$str .= '$tmpVal = ($this->getParameters()[\'templateTextValues\'] ?: ($this->global->getTemplateTextValues)($this->global))["' . $oData['name'] . '"] ?? "";';

			if ($data['defaultLangValue'] !== 'false' && $data['type'] === 'image' && Parameters::load('system.useLatteTemplateReaderDefaultLang')) {
				$str .= 'if (!$tmpVal || !$tmpVal[\'' . $oData['name'] . '_file\']) {';
				$str .= '$tmpVal = ($this->getParameters()[\'templateTextValuesLang\'][$this->global->translator->getDefaultLocale()] ?: ($this->global->getTemplateTextValues)($this->global, true))["' . $oData['name'] . '"] ?? "";';
				$str .= '}';
			}

			$str .= '$tmp = $this->global->templateText->render("' . ($data['type'] ?? 'text') . '", $tmpVal, \'' . Json::encode($data) . '\');' . PHP_EOL;
		} else {
			$str .= '$tmp = $this->global->templateText->render("' . ($data['type'] ?? 'text') . '", $templateTextValues["' . $oData['name'] . '"] ?? "", \'' . Json::encode($data) . '\');' . PHP_EOL;
		}

		if (isset($oData['toVar'])) {
			$str .= '${"' . str_replace('"', "'", $oData['toVar']) . '"} = $tmp';
		} else {
			$str .= 'echo $tmp';
		}

		return $writer->write($str);
	}

	public function macroTemplateTextR(MacroNode $node, PhpWriter $writer): string
	{
		$data = self::parseInputAttrs($node->args);
		$data = $this->prepareData($data);

		$str = 'echo \'{tt';
		foreach ($data as $k => $v) {
			$str .= ' ' . $k . '="' . $v . '"';
		}

		$str .= '}\'';

		return $writer->write($str);
	}

	private function prepareData(array $data, string $quotes = "'"): array
	{
		$re = '/{\$([^}]+)}/';
		foreach ($data as $k => $v) {
			preg_match_all($re, $v, $matches, PREG_OFFSET_CAPTURE, 0);

			foreach ($matches as $mk => $match) {
				$data[$k] = str_replace($matches[0][$mk][0], $quotes . " . $" . $matches[1][$mk][0] . " . " . $quotes, $data[$k]);
			}

			if ($v && Strings::startsWith($v, '{') && Strings::endsWith($v, '}') && Arrays::isJson($v)) {
				$data[$k] = Json::decode($v, Json::FORCE_ARRAY);
			} else if ($k === 'items') {
				$data[$k] = str_replace("'", "\'", $v);
			}
		}

		return $data;
	}

	public static function parseInputAttrs(string $str): array
	{
		$re  = '/(\w+)="(.*)"(?:\s|}$|\Z)/U';
		$arr = [];
		preg_match_all($re, $str, $matches, PREG_SET_ORDER, 0);
		foreach ($matches as $match) {
			/** @phpstan-ignore-next-line */
			if (isset($match[1]) && isset($match[2])) {
				if ($match[1] === 'multiLang' && $match[2] !== 'true') {
					continue;
				}

				$arr[$match[1]] = $match[2];
			}
		}

		return $arr;
	}
}
