<?php declare(strict_types = 1);

namespace Core\Model\Templating;

use Contributte\Translation\Translator;
use Core\FrontModule\Model\ABTest;
use Core\FrontModule\Model\Redesign;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\TemplateFactoryEvent;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Langs;
use Core\Model\Settings;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Core\Model\WebColors;
use Detection\MobileDetect;
use Nette;
use Nette\Application\LinkGenerator;
use Nette\Application\UI;
use Nette\Bridges\ApplicationLatte\LatteFactory;
use Nette\Caching\Storage;
use Nette\Http\IRequest;
use Nette\Security\User;

class TemplateFactory extends Nette\Bridges\ApplicationLatte\TemplateFactory
{
	protected array $support;

	public static ?MobileDetect $mobileDetect                  = null;
	public static ?bool         $isMultiSiteWeb                = null;
	public static array         $siteSettings                  = [];
	public static ?array        $disableCreateTemplateEventFor = null;

	/** @var Nette\DI\Container|null @inject */
	public ?Nette\DI\Container $container = null;

	public ?LinkGenerator $linkGenerator = null;

	/** @var Langs|null @inject */
	public ?Langs $langs = null;

	/** @var ImagePipe|null @inject */
	public ?ImagePipe $imagePipe = null;

	/** @var Settings|null @inject */
	public ?Settings $settings = null;

	/** @var Sites|null @inject */
	public ?Sites $sites = null;

	/** @var Translator|null @inject */
	public ?Translator $translator = null;

	/** @var WebColors|null @inject */
	public ?WebColors $webColors = null;

	/** @var ABTest|null @inject */
	public ?ABTest $abTest = null;

	/** @var Redesign|null @inject */
	public ?Redesign $redesign = null;

	/** @var EventDispatcher|null @inject */
	public ?EventDispatcher $eventDispatcher = null;

	public function __construct(
		array        $support,
		LatteFactory $latteFactory,
		IRequest     $httpRequest = null,
		User         $user = null,
		Storage      $cacheStorage = null,
		?string      $templateClass = null
	)
	{
		parent::__construct($latteFactory, $httpRequest, $user, $cacheStorage, $templateClass ?? Template::class);

		$this->support = $support;
	}

	public function createTemplate(UI\Control $control = null, string $class = null, bool $allowCreateTemplateEvent = true): UI\Template
	{
		$template = parent::createTemplate($control);

		$site = $this->sites->getCurrentSite();

		$this->linkGenerator = $this->container->hasService('application.linkGenerator')
			? $this->container->getService('application.linkGenerator')
			: null;

		$template->variables = $this->container->hasService('core.variables')
			? $this->container->getService('core.variables')
			: null;

		$template->_imagePipe    = $this->imagePipe;
		$template->linkGenerator = $this->linkGenerator;
		$template->support       = $this->support;
		$template->currentSite   = $site;
		$template->siteName      = $site->getSiteName();
		$template->adminAssets   = '/' . ASSETS_FOLDER . '/admin';
		$template->frontAssets   = '/' . ASSETS_FOLDER . '/front';
		$template->frontImages   = $template->frontAssets . '/img';
		$template->webSettings   = $this->settings->getAll();
		$template->lang          = $this->translator->getLocale();
		$template->webColors     = $this->webColors->getColors($site->getIdent());

		if (!array_key_exists($site->getIdent() . '_webSettings', self::$siteSettings)) {
			self::$siteSettings[$site->getIdent() . '_webSettings'] = $this->settings->getPrefixed($site->getIdent() . '_webSettings');
		}
		$template->siteSettings = self::$siteSettings[$site->getIdent() . '_webSettings'];

		if (self::$isMultiSiteWeb === null) {
			self::$isMultiSiteWeb = count($this->sites->getSites()) > 1;
		}
		$template->isMultiSiteWeb = self::$isMultiSiteWeb;

		$template->pshkCopyright = '<div id="pshk-copyright">&copy; by <a href="https://www.pshk.cz" target="_blank">PSHK s.r.o.</a></div>';
		$template->blocksFiles   = [SRC_DIR . '/core/src/FrontModule/Templates/blocks.latte'];
		$template->mobileDetect  = self::getMobileDetect();
		$template->abTest        = $this->abTest;
		$template->redesign      = $this->redesign;

		$controlClass = $control ? get_class($control) : '';
		if (isset(self::getDisableCreateTemplateEventFor()[$controlClass])) {
			$allowCreateTemplateEvent = false;
		}

		if ($allowCreateTemplateEvent) {
			$this->eventDispatcher->dispatch(new TemplateFactoryEvent($template), 'templateFactory::createTemplate');
		}

		return $template;
	}

	protected static function getMobileDetect(): MobileDetect
	{
		if (self::$mobileDetect === null) {
			self::$mobileDetect = new MobileDetect;
		}

		return self::$mobileDetect;
	}

	protected static function getDisableCreateTemplateEventFor(): array
	{
		if (self::$disableCreateTemplateEventFor === null) {
			self::$disableCreateTemplateEventFor = [];

			foreach (SystemConfig::load('disableCreateTemplateEventFor') as $k => $v) {
				if (is_string($v)) {
					self::$disableCreateTemplateEventFor[$v] = true;
				}
			}
		}

		return self::$disableCreateTemplateEventFor;
	}
}
