<?php declare(strict_types = 1);

namespace Core\Model;

use Core\Model\Entities\Setting;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Nette\Localization\ITranslator;
use Nette\Utils\Json;

/**
 * Class Settings
 * @package Core\Model
 */
class Settings extends BaseEntityService
{
	/** @var ITranslator @inject */
	public $translator;

	/** @var string */
	protected $entityClass = Setting::class;

	/** @var IStorage */
	protected $cacheStorage;

	/** @var Cache */
	protected $cache;

	public function __construct(IStorage $storage)
	{
		$this->cacheStorage = $storage;
	}

	/**
	 * @return Cache
	 */
	public function getCache(): Cache
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, 'coreSettings');

		return $this->cache;
	}

	/**
	 * @param string $key
	 *
	 * @return string
	 */
	public function get(string $key): string
	{
		return $this->getAll()[$key] ?? '';
	}

	/**
	 * @return array
	 */
	public function getAll(): array
	{
		$cacheKey = 'coreSettingsAll/' . $this->translator->getLocale();

		return $this->getCache()->load($cacheKey, function(&$dep) {
			$dep    = [Cache::TAGS => ['coreSettings'], Cache::EXPIRE => '1 month'];
			$result = [];
			foreach ($this->getAllRaw() as $k => $v) {
				if (Arrays::isJson($v)) {
					$result[$k] = Json::decode($v, true)[$this->translator->getLocale()] ?? null;
				} else {
					$result[$k] = $v;
				}
			}

			return $result;
		});
	}

	public function getAllRaw(): array
	{
		$result = [];
		foreach ($this->getEr()->createQueryBuilder('s')->getQuery()->getScalarResult() as $v)
			$result[$v['s_key']] = $v['s_value'];

		return $result;
	}

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return Setting
	 * @throws \Exception
	 */
	public function save(string $key, string $value): Setting
	{
		$entity = $this->getEr()->find($key);
		if (!$entity)
			$entity = new Setting($key, $value);
		else
			$entity->value = $value;
		$this->em->persist($entity)->flush();

		return $entity;
	}

	/**
	 * @param array $data
	 *
	 * @throws \Exception
	 */
	public function saveMulti(array $data): void
	{
		$exist = $this->getEr()->createQueryBuilder('s', 's.key')->getQuery()->getResult();

		foreach ($data as $k => $v) {
			if (isset($exist[$k]))
				$entity = $exist[$k];
			else
				$entity = new Setting($k, $v);
			$entity->value = $v;
			$this->em->persist($entity);
		}

		$this->em->flush();
	}

	public function clearCache(): void
	{
		$this->getCache()->clean([Cache::TAGS => ['coreSettings']]);
	}
}
