<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\UI\BaseControl;

class Navigation extends BaseControl
{
	/** @var array */
	protected $data;

	/** @var DaoNavigationItem[][] */
	protected array $sections = [
		'top'         => [
			'title' => null,
			'items' => [],
		],
		'eshop'       => [
			'title' => 'core.adminNavigation.eshop',
			'items' => [],
		],
		'pageContent' => [
			'title' => 'core.adminNavigation.pageContent',
			'items' => [],
		],
		'settings'    => [
			'title' => 'core.adminNavigation.settings',
			'items' => [],
		],
	];

	public function render(): void
	{
		$this->template->sections = $this->sections;
		$this->template->render($this->getTemplateFile());
	}

	public function setActiveNav(): void
	{
		$this->setDataToSections($this->data);
		$presenter = $this->getPresenter();

		$setCurrent = function($navs) use (&$setCurrent, $presenter) {
			/** @var DaoNavigationItem[] $navs */
			foreach ($navs as $nav) {
				$link           = is_array($nav->link) ? $nav->link[0] : $nav->link;
				$nav->isCurrent = $nav->link && (($presenter->isLinkCurrent($link, is_array($nav->link) ? $nav->link[1] : [])) || (!is_array($nav->link) && $presenter->isModuleCurrent($link)));

				if ($nav->isCurrent)
					$presenter->activeNavigation = $nav;

				if ($nav->childs)
					$setCurrent($nav->childs);
			}
		};

		foreach ($this->sections as $section => $navs) {
			$setCurrent($navs['items']);
		}
	}

	/**
	 * Data z konfigu
	 *
	 * @param array $data
	 */
	public function setData(array $data): void
	{
		$this->data = $data;
	}

	/**
	 * @return array
	 */
	public function getData(): array
	{
		return $this->data;
	}

	/**
	 * Nastaví data z konfigu do DAO
	 *
	 * @param array $data
	 */
	public function setDataToSections(array $data): void
	{
		foreach ($data as $section => $rows) {
			foreach ($rows as $row) {
				if (!isset($row['title']))
					continue;

				$nav = new DaoNavigationItem($row);
				if ($this->checkAcl($nav))
					$this->sections[$section]['items'][] = $nav;
			}
		}

		foreach ($this->sections as $section => $v) {
			$this->sort($this->sections[$section]['items']);
		}
	}

	/**
	 * Zkontroluje jestli má uživatel přístup. Podařazené položky rovnou maže
	 *
	 * @param DaoNavigationItem $nav
	 *
	 * @return bool
	 */
	protected function checkAcl(DaoNavigationItem $nav): bool
	{
		$user = $this->getPresenter()->getUser();

		if ($nav->securityConfig === false)
			return false;

		$allowed = $nav->security ? false : true;

		if (!$allowed)
			foreach ($nav->security as $m => $ps) {
				if ($user->isAllowed($m, $ps)) {
					$allowed = true;
					break;
				}
			}

		foreach ($nav->childs as $k => $c) {
			if (!$this->checkAcl($c))
				$nav->childs[$k]->hidden = true;
		}

		return $allowed;
	}

	/**
	 * Seřadí položky podle order
	 *
	 * @param DaoNavigationItem[] $data
	 *
	 * @return DaoNavigationItem[]
	 */
	protected function sort(&$data): array
	{
		if (is_array($data)) {
			usort($data, function($a, $b) {
				return $a->order - $b->order;
			});

			foreach ($data as &$v)
				if ($v->childs)
					$this->sort($v->childs);
		}

		return $data;
	}
}
