<?php declare(strict_types = 1);

namespace Core\AdminModule\Components;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\DataToFormInputs;
use Core\Model\Settings;
use Core\Model\SystemConfig;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;

abstract class SettingsForm extends BaseControl
{
	/** @var SystemConfig @inject */
	public $systemConfig;

	/** @var DataToFormInputs @inject */
	public $dataToFormInputs;

	/** @var Settings @inject */
	public $settings;

	/** @var string */
	protected $settingsNamespace;

	/** @var string */
	public $inputNamePrefix = '';

	/** @var string */
	protected $templateFile;

	public function __construct(string $namespace)
	{
		$this->settingsNamespace = $namespace;
		$this->templateFile      = __DIR__ . '/SettingsForm.latte';
	}

	public function render()
	{
		$this->template->thisForm = $this['form'];
		$this->template->setFile($this->templateFile);
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addComponent(new BaseContainer(), 'component');
		$form->addSubmit('submit', 'default.save');

		$component = $form['component'];
		$parsed    = $this->dataToFormInputs->parseData($component, $this->getSystemConfigData());

		$form->onSuccess[] = [$this, 'formSuccess'];

		$this->template->componentStructure = $parsed['structure'];

		$defaults    = [];
		$settingsRaw = $this->settings->getAllRaw();
		foreach ($this->getSystemConfigData() as $v) {
			if (!isset($v['name']))
				continue;

			$d = $settingsRaw[$v['name']] ?? null;
			if (Arrays::isJson($d) && !is_numeric($d))
				$d = Json::decode($d, Json::FORCE_ARRAY);

			if ($d)
				$defaults[$v['name']] = $d;
		}
		$this->dataToFormInputs->setDefaults($form['component'], $defaults);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$this->em->beginTransaction();
		try {
			$presenter = $this->getPresenter();

			$vals = [];
			foreach ($form->getComponent('component')->getComponents() as $c) {
				if ($c->getIsMultiLanguage())
					$vals[$c->getName()] = Json::encode($c->getValue());
				else
					$vals[$c->getName()] = $c->getValue();
			}

			$this->settings->saveMulti($vals);

			$presenter->flashMessageSuccess('default.saved');
			$presenter->redrawControl('flashes');

			$this->em->commit();
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}
		$this->settings->clearCache();
		$this->redrawControl('form');

		return true;
	}

	public function getDataForInputs(): array
	{
		return $this->systemConfig->get($this->settingsNamespace, []);
	}

	/**
	 * @return array
	 */
	public function getSystemConfigData(): array
	{
		$result          = [];
		$inputNamePrefix = $this->inputNamePrefix ? $this->inputNamePrefix . '_' : '';
		foreach ($this->getDataForInputs() as $v) {
			if (isset($v['showIf']) && !$v['showIf'])
				continue;

			$arr = $v['name'];
			if (!is_array($v['name']))
				$arr = [$v['name']];

			foreach ($arr as $name) {
				$tmp = $v;
				if (is_array($v['name']))
					$tmp['title'][0] = $v['title'][0] . '.' . $name;
				if ($this->settingsNamespace !== 'globalData')
					$tmp['name'] = $this->settingsNamespace . ucfirst($name);
				$tmp['name'] = $inputNamePrefix . $tmp['name'];
				$result[]    = $tmp;
			}
		}

		return $result;
	}
}
