<?php declare(strict_types = 1);

namespace Core\Model;

use Core\Model\Entities\Setting;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Nette\Localization\ITranslator;
use Nette\Utils\Json;

/**
 * Class Settings
 * @package Core\Model
 */
class Settings extends BaseEntityService
{
	/** @var ITranslator @inject */
	public $translator;

	/** @var string */
	protected $entityClass = Setting::class;

	/** @var IStorage */
	protected $cacheStorage;

	/** @var Cache */
	protected $cache;

	protected ?array $cLoaded = null;

	public function __construct(IStorage $storage)
	{
		$this->cacheStorage = $storage;
	}

	/**
	 * @return Cache
	 */
	public function getCache(): Cache
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, 'coreSettings');

		return $this->cache;
	}

	/**
	 * @param string $key
	 * @param mixed  $default
	 * @param bool   $useCurrentLanguage
	 *
	 * @return string|array
	 */
	public function get(string $key, $default = '', bool $useCurrentLanguage = true)
	{
		return $this->getAll($useCurrentLanguage)[$key] ?? $default;
	}

	/**
	 * @param bool $useCurrentLanguage
	 *
	 * @return array
	 */
	public function getAll(bool $useCurrentLanguage = true): array
	{
		$cKey = $useCurrentLanguage ? $this->translator->getLocale() : 'not';
		if ($this->cLoaded[$cKey] === null) {
			$cacheKey = 'coreSettingsAll/' . $cKey;

			$this->cLoaded[$cKey] = $this->getCache()->load($cacheKey, function(&$dep) use ($useCurrentLanguage) {
				$dep    = [Cache::TAGS => ['coreSettings'], Cache::EXPIRE => '2 weeks'];
				$result = [];

				foreach ($this->getAllRaw() as $k => $v) {
					if (Arrays::isJson($v) && !is_numeric($v)) {
						$tmp = Json::decode($v, Json::FORCE_ARRAY);

						if ($useCurrentLanguage && isset($tmp[$this->translator->getLocale()])) {
							$result[$k] = $tmp[$this->translator->getLocale()];
						} else {
							$result[$k] = $tmp;
						}
					} else {
						$result[$k] = $v;
					}
				}

				return $result;
			});
		}

		return $this->cLoaded[$cKey];
	}

	public function getPrefixed(string $prefix): array
	{
		$result = [];
		$length = strlen($prefix);

		foreach ($this->getAll() as $k => $v) {
			if (strpos($k, $prefix) === 0) {
				$k          = lcfirst(substr($k, $length));
				$result[$k] = $v;
			}
		}

		return $result;
	}

	public function getAllRaw(): array
	{
		$result = [];
		foreach ($this->getEr()->createQueryBuilder('s')->getQuery()->getScalarResult() as $v) {
			if ($v['s_value'] === '[]')
				$v['s_value'] = [];
			$result[$v['s_key']] = $v['s_value'];
		}

		return $result;
	}

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return Setting
	 * @throws \Exception
	 */
	public function save(string $key, string $value): Setting
	{
		$entity = $this->getEr()->find($key);
		if (!$entity)
			$entity = new Setting($key, $value);
		else
			$entity->value = $value;
		$this->em->persist($entity)->flush();

		return $entity;
	}

	/**
	 * @param array $data
	 *
	 * @throws \Exception
	 */
	public function saveMulti(array $data): void
	{
		$exist = $this->getEr()->createQueryBuilder('s', 's.key')->getQuery()->getResult();

		foreach ($data as $k => $v) {
			if (isset($exist[$k]))
				$entity = $exist[$k];
			else
				$entity = new Setting($k, $v);
			$entity->setValue($v);
			$this->em->persist($entity);
		}

		$this->em->flush();
	}

	public function clearCache(): void
	{
		foreach ($this->translator->getAvailableLocales() as $lang)
			$this->getCache()->remove('coreSettingsAll/' . explode('_', $lang)[0]);

		$this->getCache()->clean([Cache::TAGS => ['coreSettings']]);
		$this->cLoaded = null;
	}
}
