<?php declare(strict_types = 1);

namespace Core\Components\Navigation;

use Core\Model\UI\BaseControl;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;

class Navigation extends BaseControl
{
	protected ArrayHash $data;
	public array        $templates = [];

	public function __construct(
		protected Request $httpRequest,
	)
	{
		$this->data = new ArrayHash();
	}

	public function render(array|string|null $group, string $template = 'default'): void
	{
		if (is_array($group)) {
			$data = $group;
		} else if (!isset($this->data->{$group})) {
			return;
		} else {
			/** @var DaoNavigationItem[] $data */
			$data = $this->data->{$group};

			$this->sort($data);
			$this->checkAcl($data);
			$this->checkIsCurrent($data);
		}

		$this->template->data = $data;
		$this->template->setFile($this->templates[$template] ?? $this->getTemplateFile($template));

		$this->template->render();
	}

	/**
	 * Zjištění jestli uživatel může odkaz vidět
	 *
	 * @param DaoNavigationItem[] $data
	 */
	private function checkAcl(&$data): void
	{
		$user = $this->presenter->getUser();

		foreach ($data as $k => $v) {
			$allowed = $v->security ? false : true;

			if (!$allowed) {
				foreach ($v->security as $m => $ps) {
					if ($user->isAllowed($m, $ps)) {
						$allowed = true;
						break;
					}
				}
			}

			if (!$allowed) {
				unset($data[$k]);
			}

			if ($v->childs) {
				$this->checkAcl($v->childs);
			}
		}
	}

	/**
	 * Zjištění jestli je odkaz aktivní
	 *
	 * @param DaoNavigationItem[] $data
	 */
	private function checkIsCurrent(array &$data): void
	{
		$fn = function(&$data) use (&$fn) {
			foreach ($data as $nav) {
				/** @var DaoNavigationItem $nav */
				if ($nav->isHomepage) {
					continue;
				}

				$path      = explode('/', $this->httpRequest->getUrl()->path);
				$isCurrent = false;

				do {
					if (implode('/', $path) == $nav->link && $nav->link !== null && $nav->link !== '') {
						$isCurrent = true;
						break;
					}

					array_pop($path);
				} while (!empty($path));

				if ($isCurrent) {
					$nav->isCurrent = true;
				}

				if ($nav->childs) {
					$fn($nav->childs);

					foreach ($nav->childs as $child) {
						if ($child->isCurrent) {
							$nav->isCurrent = true;
							break;
						}
					}
				}
			}
		};

		$fn($data);
	}

	public function setData(?array $data): void
	{
		if (empty($data)) {
			return;
		}

		foreach ($data as $group => $navs) {
			if (!isset($this->data->$group)) {
				$this->data->$group = [];
			}

			foreach ($navs as $navName => $nav) {
				$this->data->$group[$navName] = $nav instanceof DaoNavigationItem ? $nav : new DaoNavigationItem($nav);
			}
		}
	}

	/**
	 * @param array|object $data
	 */
	private function sort(&$data): array|object
	{
		if (is_array($data)) {
			usort($data, fn($a, $b) => $a->order - $b->order);

			foreach ($data as &$v) {
				if ($v->childs) {
					$this->sort($v->childs);
				}
			}
		}

		return $data;
	}
}
