<?php declare(strict_types = 1);

namespace Core\Console\Database;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Exception;
use Nette\DI\Attributes\Inject;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class ClearMigrations extends AbstractCommand
{
	#[Inject]
	public EntityManagerDecorator $em;

	protected function configure(): void
	{
		$this->setName('database:clear-migrations')->setDescription('Clear migrations');
	}

	protected function executeSchemaCommand(
		InputInterface                $input,
		OutputInterface               $output,
		Doctrine\ORM\Tools\SchemaTool $schemaTool,
		array                         $metadatas,
		SymfonyStyle                  $ui,
	)
	{
		$conn = $this->em->getConnection();

		$rename = [
			'dynamicmodule' => 'DynamicModule',
			'eshopcatalog'  => 'EshopCatalog',
			'eshoporders'   => 'EshopOrders',
		];

		$output->writeln('CLEAR MIGRATIONS');
		try {
			$migrations = [];
			foreach ($conn->fetchAllAssociative('SELECT version FROM doctrine_migrations') as $row) {
				$migrations[$row['version']] = $row['version'];
			}

			$allFiles = [];
			foreach (glob(ROOT_DIR . '/vendor/pshkcms/*/src/Migrations/*/*/*.php') ?: [] as $file) {
				$tmp   = explode('pshkcms/', $file);
				$parts = explode('/', $tmp[1]);

				$module   = $rename[$parts[0]] ?? $parts[0];
				$module   = ucfirst($module);
				$basename = basename($file);
				$number   = (int) filter_var($basename, FILTER_SANITIZE_NUMBER_INT);
				$version  = 'Version' . $number;

				$allFiles[$module][] = [
					'basename' => $basename,
					'number'   => $number,
					'version'  => $version,
				];

				$newVersion = $module . "\\Migrations\\Version" . $number;
				if (isset($migrations[$number])) {
					unset($migrations[$number]);
					$conn->executeQuery("UPDATE doctrine_migrations SET version = ? WHERE version = ?", [
						$newVersion, $number,
					]);
				} else if (isset($migrations['Migrations\\' . $version])) {
					unset($migrations['Migrations\\' . $version]);
					$conn->executeQuery("UPDATE doctrine_migrations SET version = ? WHERE version = ?", [
						$newVersion, 'Migrations\\' . $version,
					]);
				} else if (isset($migrations[$newVersion])) {
					unset($migrations[$newVersion]);
				}
			}

			foreach ($migrations as $row) {
				$conn->executeQuery("DELETE FROM doctrine_migrations WHERE version = ?", [$row]);
			}

			$output->writeLn('<info>[OK] - cleared</info>');
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
