<?php declare(strict_types = 1);

namespace Core\Model\Helpers;

use Contributte\Translation\Translator;
use Core\Model\TemplateReader\TemplateTextTypesCollection;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\HiddenInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Exception;
use Nette\Forms\Controls\BaseControl;
use Nette\Utils\Json;

class DataToFormInputs
{
	public function __construct(
		protected TemplateTextTypesCollection $typesCollection,
		protected Translator                  $translator,
	)
	{
	}

	public function parseData(BaseContainer $container, array $data, array $texts = []): array
	{
		$structure    = ['sections' => []];
		$sectionNames = [];
		$groupNames   = [];

		foreach ($data as $v) {
			$item = $this->typesCollection->getItemByType($v['type'] ?? 'textarea');

			if ($item && $v['name']) {
				if ($v['title']) {
					$v['title'] = $this->translator->translate(...$v['title']);
					$item->setTitle($v['title']);
				}

				$item->setName($v['name']);

				/** @var array<string, array> $s */
				$s = &$structure;

				if (isset($v['items']) && is_string($v['items'])) {
					$v['items'] = str_replace("'", '"', $v['items']);

					if (Arrays::isJson($v['items'])) {
						$v['items'] = Json::decode($v['items'], Json::FORCE_ARRAY);
					} else {
						$v['items'] = explode(',', $v['items']);
					}
				}

				if (isset($v['section'])) {
					if (isset($v['sectionName'])) {
						$sectionNames[$v['section']] = $this->translator->translate(...(array) $v['sectionName']);
					}

					if (!isset($s['sections'][$v['section']])) {
						$s['sections'][$v['section']] = [];
					}

					$s = &$s['sections'][$v['section']];
				}

				if (isset($v['group'])) {
					if (isset($v['groupName'])) {
						$groupNames[$v['group']] = $this->translator->translate(...(array) $v['groupName']);
					}

					if (!isset($s[$v['group']])) {
						$s[$v['group']] = [];
					}

					$s = &$s[$v['group']];
				}

				if (isset($v['default'])) {
					$item->setDefault($v['default']);
				}

				$s[] = $v['name'];
				if (!$container->getComponent($v['name'], false)) {
					$item->loadContainer($container, $v, $texts);
				}

				if (isset($v['default'])) {
					$container->getComponent($v['name'])->setDefaultValue($v['default']);
				}

				if (isset($v['description'])) {
					if (is_string($v['description'])) {
						$v['description'] = str_replace("'", '"', $v['description']);
						if (Arrays::isJson($v['description'])) {
							$v['description'] = Json::decode($v['description'], Json::FORCE_ARRAY);
						} else {
							$v['description'] = [trim($v['description'], '[]\'')];
						}
					}

					$component = $container->getComponent($v['name'], false);
					if ($component && is_object($component) && method_exists($component, 'setDescription')) {
						$component->setDescription($this->translator->translate(...$v['description']));
					}
				}
			}
		}

		if (count($structure['sections']) > 0) {
			foreach ($structure['sections'] as $sk => $sv) {
				if (array_key_exists($sk, $sectionNames)) {
					$structure['sections'][$sectionNames[$sk]] = $sv;
					unset($structure['sections'][$sk]);
				}
			}

			foreach ($structure['sections'] as $sk => $sv) {
				if (is_array($sv)) {
					foreach ($sv as $gk => $gv) {
						if (array_key_exists($gk, $groupNames)) {
							$structure['sections'][$sk][$groupNames[$gk]] = $gv;
							unset($structure['sections'][$sk][$gk]);
						}
					}
				}
			}
		}

		return [
			'structure' => $structure,
		];
	}

	public function setDefaults(BaseContainer $container, array $data): void
	{
		foreach ($container->getComponents() as $k => $c) {
			$value = $data[$k] ?? null;

			if ($c instanceof BaseContainer === false && method_exists(
					$c,
					'getIsMultiLanguage',
				) && $c->getIsMultiLanguage()) {
				if (is_string($value)) {
					try {
						$value = Json::decode($value, Json::FORCE_ARRAY);
					} catch (Exception) {
					}
				}

				try {
					if (is_string($value)
						|| (is_array($value) && !isset($value[$this->translator->getDefaultLocale()]))
						|| (is_object($value) && !isset($value->{$this->translator->getDefaultLocale()}))) {
						$valueTmp = [];
						foreach ($this->translator->getLocalesWhitelist() as $l) {
							$valueTmp[$l] = $value;
						}

						$value = $valueTmp;
					}
				} catch (Exception) {
				}
			}

			if (!$value) {
				continue;
			}

			if ($c instanceof SelectInput) {
				if (array_key_exists($value, $c->getItems())) {
					$c->setDefaultValue($value);
				}
			} else if ($c instanceof CheckboxListInput) {
				$tmp = [];
				foreach ($value as $v) {
					if (array_key_exists($v, $c->getItems())) {
						$tmp[] = $v;
					}
				}

				$c->setDefaultValue($tmp);
			} else if ($c instanceof HiddenInput) {
				$c->setValue($value);
			} else if ($c instanceof BaseControl) {
				$c->setDefaultValue($value);
			} else if ($c instanceof BaseContainer && (is_array($value) || is_object($value))) {
				$this->setDefaults($c, (array) $value);
			}
		}
	}
}
