<?php declare(strict_types = 1);

namespace Core\Model\Mailing;

use Core\Model\Templating\Template;
use LogicException;
use Nette\Utils\Strings;
use function str_starts_with;

abstract class AbstractTemplateFactory
{
	protected array $defaults = [
		'layout' => '@default',
	];

	protected array $config = [
		'layout' => '@default',
	];

	/**
	 * GETTERS/SETTERS *********************************************************
	 */

	public function setDefaults(array $defaults): void
	{
		$this->defaults = $defaults;
	}

	public function setDefaultsItem(string $key, mixed $value): void
	{
		$this->defaults[$key] = $value;
	}

	public function setConfig(array $config): void
	{
		$this->config = $config;
	}

	public function setConfigItem(string $key, mixed $value): void
	{
		$this->config[$key] = $value;
	}

	/**
	 * HELPERS *****************************************************************
	 */

	protected function prepare(Template $template): Template
	{
		$template = $this->prepareDefaults($template);
		$template = $this->prepareConfig($template);

		return $template;
	}

	protected function prepareDefaults(Template $template): Template
	{
		// Layout
		if ($this->defaults['layout']) {
			$this->defaults['layout'] = $this->prepareLayout($this->defaults['layout']);
		}

		// Append defaults to template
		$template->add('_defaults', (object) $this->defaults);

		return $template;
	}

	protected function prepareConfig(Template $template): Template
	{
		// Layout
		if ($this->config['layout']) {
			$this->config['layout'] = $this->prepareLayout($this->config['layout']);
		}

		// Append defaults to template
		$template->add('_config', (object) $this->config);

		return $template;
	}

	protected function prepareLayout(string $layout): string
	{
		if (str_starts_with($layout, '@')) {
			$layout = __DIR__ . '/layouts/' . $layout;
		}

		$layout = Strings::replace($layout, '#.latte$#', '') . '.latte';

		if (!file_exists($layout)) {
			throw new LogicException(sprintf('Layout file "%s" not found', $layout));
		}

		return $layout;
	}

}
