<?php declare(strict_types = 1);

namespace Core\Model\Mailing;

use Core\Model\Templating\Template;
use Latte\Engine;
use Nette\Mail\Mailer;
use Nette\Mail\Message;

class MailBuilder
{
	public ?Message  $message  = null;
	public ?Template $template = null;

	public function __construct(
		public Mailer $sender,
		Message       $message = null,
	)
	{
		$this->message = $message ?: new Message();
	}

	public function setMessage(Message $message): self
	{
		$this->message = $message;

		return $this;
	}

	/**
	 * TEMPLATE API ************************************************************
	 */

	public function setTemplate(Template $template): void
	{
		$this->template = clone $template;
	}

	public function getTemplate(): Template
	{
		return $this->template ?: $this->template = new Template(new Engine());
	}

	public function setParameters(array $parameters): self
	{
		$this->getTemplate()->setParameters($parameters);

		return $this;
	}

	public function setTemplateFile(string $file): self
	{
		$this->getTemplate()->setFile($file);

		return $this;
	}

	/**
	 * MESSAGE API *************************************************************
	 */

	public function getMessage(): Message
	{
		if (!$this->message) {
			$this->message = new Message;
		}

		return $this->message;
	}

	public function addTo(string $email, ?string $name = null): self
	{
		$this->message->addTo($email, $name);

		return $this;
	}

	public function addBcc(string $email, ?string $name = null): self
	{
		$this->message->addBcc($email, $name);

		return $this;
	}

	public function addCc(string $email, ?string $name = null): self
	{
		$this->message->addCc($email, $name);

		return $this;
	}

	public function setSubject(string $subject): self
	{
		$this->message->setSubject($subject);

		return $this;
	}

	public function addReplyTo(string $email, ?string $name = null): self
	{
		$this->message->addReplyTo($email, $name);

		return $this;
	}

	public function setFrom(string $from, ?string $fromName = null): self
	{
		$this->message->setFrom($from, $fromName);

		return $this;
	}

	public function call(callable $callback): self
	{
		$callback($this->message, $this->template);

		return $this;
	}

	/**
	 * SENDER API **************************************************************
	 */

	public function send(): void
	{
		// Create message
		$message = $this->getMessage();

		// Create template
		$template = $this->getTemplate();
		$template->add('_mail', $message);

		$message->setHtmlBody($template->renderToString());

		// Send message
		$this->sender->send($message);
	}

}
