<?php declare(strict_types = 1);

namespace Core\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\Setting;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Exception;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Utils\Json;

class Settings extends BaseEntityService
{
	/** @var Translator @inject */
	public $translator;

	/** @var string */
	protected $entityClass = Setting::class;

	/** @var Storage */
	protected $cacheStorage;

	/** @var Cache */
	protected $cache;

	protected ?array $cLoaded = null;

	public function __construct(Storage $storage)
	{
		$this->cacheStorage = $storage;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, 'coreSettings');
		}

		return $this->cache;
	}

	/**
	 * @param mixed $key
	 * @param mixed $default
	 *
	 * @return mixed|null
	 */
	public function get($key, $default = '', bool $useCurrentLanguage = true)
	{
		return $this->getAll($useCurrentLanguage)[$key] ?? $default;
	}

	public function getAll(bool $useCurrentLanguage = true): array
	{
		$cKey = $useCurrentLanguage ? $this->translator->getLocale() : 'not';
		if ($this->cLoaded === null || $this->cLoaded[$cKey] === null) {
			$cacheKey = 'coreSettingsAll/' . $cKey;

			$this->cLoaded[$cKey] = $this->getCache()->load($cacheKey, function(&$dep) use ($useCurrentLanguage) {
				$dep    = [Cache::Tags => ['coreSettings'], Cache::Expire => '2 weeks'];
				$result = [];

				foreach ($this->getAllRaw() as $k => $v) {
					if (is_string($v) && !is_numeric($v) && Arrays::isJson($v)) {
						$tmp = Json::decode($v, Json::FORCE_ARRAY);

						if ($useCurrentLanguage && isset($tmp[$this->translator->getLocale()])) {
							$result[$k] = $tmp[$this->translator->getLocale()];
						} else {
							$result[$k] = $tmp;
						}
					} else {
						$result[$k] = $v;
					}
				}

				return $result;
			});
		}

		return $this->cLoaded[$cKey];
	}

	public function getPrefixed(string $prefix): array
	{
		$result = [];
		$length = strlen($prefix);

		foreach ($this->getAll() as $k => $v) {
			if (str_starts_with($k, $prefix)) {
				$k          = lcfirst(substr($k, $length));
				$result[$k] = $v;
			}
		}

		return $result;
	}

	public function getAllRaw(): array
	{
		$result = [];
		foreach ($this->getEr()->createQueryBuilder('s')->getQuery()->getScalarResult() as $v) {
			if ($v['s_value'] === '[]') {
				$v['s_value'] = [];
			} else if (Arrays::isJson($v['s_value'])) {
				$v['s_value'] = Json::decode($v['s_value'], Json::FORCE_ARRAY);
			}
			$result[$v['s_key']] = $v['s_value'];
		}

		return $result;
	}

	/**
	 *
	 * @throws Exception
	 */
	public function save(string $key, string $value): Setting
	{
		/** @var Setting|null $entity */
		$entity = $this->getEr()->find($key);
		if (!$entity) {
			$entity = new Setting($key, $value);
		} else {
			$entity->value = $value;
		}

		$this->em->persist($entity)->flush();

		return $entity;
	}

	/**
	 * @throws Exception
	 */
	public function saveMulti(array $data): void
	{
		$exist = $this->getEr()->createQueryBuilder('s', 's.key')->getQuery()->getResult();

		foreach ($data as $k => $v) {
			if (isset($exist[$k])) {
				$entity = $exist[$k];
			} else {
				$entity = new Setting($k, $v);
			}

			$entity->setValue($v);
			$this->em->persist($entity);
		}

		$this->em->flush();
	}

	public function clearCache(): void
	{
		foreach ($this->translator->getAvailableLocales() as $lang) {
			$this->getCache()->remove('coreSettingsAll/' . explode('_', $lang)[0]);
		}

		$this->getCache()->clean([Cache::Tags => ['coreSettings']]);
		$this->cLoaded = null;
	}
}
