<?php declare(strict_types = 1);

namespace Core\Model\Templating;

use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\TemplateFactoryEvent;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Langs;
use Core\Model\Settings;
use Core\Model\Sites;
use Core\Model\WebColors;
use Detection\MobileDetect;
use Nette;
use Nette\Application\LinkGenerator;
use Nette\Application\UI;
use Nette\Bridges\ApplicationLatte\LatteFactory;
use Nette\Caching\Storage;
use Nette\DI\Attributes\Inject;
use Nette\Http\IRequest;
use Nette\Security\User;

class TemplateFactory extends Nette\Bridges\ApplicationLatte\TemplateFactory
{
	#[Inject]
	public Nette\DI\Container $container;

	#[Inject]
	public LinkGenerator $linkGenerator;

	#[Inject]
	public Langs $langs;

	#[Inject]
	public ImagePipe $imagePipe;

	#[Inject]
	public Settings $settings;

	#[Inject]
	public Sites $sites;

	#[Inject]
	public Translator $translator;

	#[Inject]
	public WebColors $webColors;

	#[Inject]
	public EventDispatcher $eventDispatcher;

	public function __construct(
		protected array $support,
		LatteFactory    $latteFactory,
		IRequest        $httpRequest = null,
		User            $user = null,
		Storage         $cacheStorage = null,
		?string         $templateClass = null,
	)
	{
		parent::__construct($latteFactory, $httpRequest, $user, $cacheStorage, $templateClass ?? Template::class);
	}

	public function createTemplate(
		UI\Control $control = null,
		string     $class = null,
		bool       $allowCreateTemplateEvent = true,
	): UI\Template
	{
		$template = parent::createTemplate($control);

		$site = $this->sites->getCurrentSite();

		$this->linkGenerator = $this->container->getService('application.linkGenerator');

		$template->_imagePipe     = $this->imagePipe;
		$template->linkGenerator  = $this->linkGenerator;
		$template->support        = $this->support;
		$template->currentSite    = $site;
		$template->siteName       = $site->getSiteName();
		$template->adminAssets    = '/' . ASSETS_FOLDER . '/admin';
		$template->frontAssets    = '/' . ASSETS_FOLDER . '/front';
		$template->frontImages    = $template->frontAssets . '/img';
		$template->webSettings    = $this->settings->getAll();
		$template->siteSettings   = $this->settings->getPrefixed($site->getIdent() . '_webSettings');
		$template->lang           = $this->translator->getLocale();
		$template->webColors      = $this->webColors->getColors($site->getIdent());
		$template->isMultiSiteWeb = count($this->sites->getSites()) > 1 ? true : false;
		$template->pshkCopyright  = '<div id="pshk-copyright">&copy; by <a href="https://www.pshk.cz" target="_blank">PSHK s.r.o.</a></div>';
		$template->blocksFiles    = [SRC_DIR . '/core/src/FrontModule/Templates/blocks.latte'];
		$template->mobileDetect   = new MobileDetect;

		if ($allowCreateTemplateEvent) {
			$this->eventDispatcher->dispatch(new TemplateFactoryEvent($template), 'templateFactory::createTemplate');
		}

		return $template;
	}
}
