<?php declare(strict_types = 1);

namespace Core\Model\UI\DataGrid;

use Core\Model\UI\Form\BaseForm;
use Nette\Application\UI\Form;
use Nette\Forms\Container;
use Nette\Forms\Controls\SubmitButton;
use Ublaboo\DataGrid\DataGrid;
use Ublaboo\DataGrid\Exception\DataGridGroupActionException;
use Ublaboo\DataGrid\GroupAction\GroupAction;
use Ublaboo\DataGrid\GroupAction\GroupActionCollection;
use Ublaboo\DataGrid\GroupAction\GroupMultiSelectAction;
use Ublaboo\DataGrid\GroupAction\GroupSelectAction;
use Ublaboo\DataGrid\GroupAction\GroupTextAction;
use Ublaboo\DataGrid\GroupAction\GroupTextareaAction;

class CustomGroupActionCollection extends GroupActionCollection
{
	final public const ID_ATTRIBUTE_PREFIX = 'group_action_custom_item_';

	/**
	 * @var GroupAction[]
	 */
	protected $groupActionsCustom = [];

	public function __construct(DataGrid $datagrid)
	{
		$this->datagrid = $datagrid;
	}

	public function addToFormContainer(Container $container): void
	{
		/** @var BaseForm $form */
		$form         = $container->lookup(BaseForm::class);
		$translator   = $form->getTranslator();
		$main_options = [];

		/**
		 * First foreach for filling "main" select
		 */
		foreach ($this->groupActionsCustom as $id => $action) {
			$main_options[$id] = $action->getTitle();
		}

		$container->addSelect('groupActionCustom', '', $main_options)
			->setPrompt('ublaboo_datagrid.choose');

		/**
		 * Second for creating select for each "sub"-action
		 */
		foreach ($this->groupActionsCustom as $id => $action) {
			$control = null;

			if ($action instanceof GroupSelectAction) {
				if ($action->hasOptions()) {
					if ($action instanceof GroupMultiSelectAction) {
						$control = $container->addMultiSelect($id, '', $action->getOptions());
						$control->setHtmlAttribute('data-datagrid-multiselect-id', static::ID_ATTRIBUTE_PREFIX . $id);
						$control->setHtmlAttribute('data-style', 'hidden');
						$control->setHtmlAttribute('data-selected-icon-check', DataGrid::$iconPrefix . 'check');
					} else {
						$control = $container->addSelect((string) $id, $action->getTitle(), $action->getOptions());
						$control->setPrompt('ublaboo_datagrid.choose');
					}

					$control->setHtmlAttribute('id', static::ID_ATTRIBUTE_PREFIX . $id);
					$control->setHtmlAttribute('data-grid-group-action-custom-select', '1');
				}
			} else if ($action instanceof GroupTextAction) {
				$control = $container->addText($id, '');

				$control->setHtmlAttribute('id', static::ID_ATTRIBUTE_PREFIX . $id)
					->addConditionOn($container->getComponent('groupActionCustom'), Form::EQUAL, $id)
					->setRequired($translator->translate('ublaboo_datagrid.choose_input_required'))
					->endCondition();
			} else if ($action instanceof GroupTextareaAction) {
				$control = $container->addTextarea($id, '');

				$control->setHtmlAttribute('id', static::ID_ATTRIBUTE_PREFIX . $id)
					->addConditionOn($container->getComponent('groupActionCustom'), Form::EQUAL, $id)
					->setRequired($translator->translate('ublaboo_datagrid.choose_input_required'));
			}

			if ($control) {
				/**
				 * User may set a class to the form control
				 */
				if ($class = $action->getClass()) {
					$control->setHtmlAttribute('class', $class);
				}

				/**
				 * User may set additional attribtues to the form control
				 */
				foreach ($action->getAttributes() as $name => $value) {
					$control->setHtmlAttribute($name, $value);
				}
			}

			$container->addSubmit('submit_' . $id, 'ublaboo_datagrid.execute')
				->setValidationScope([$container])
				->setHtmlAttribute('id', strtolower($this->datagrid->getName()) . 'groupActionCustomSubmit');
		}

		$form->onSubmit[] = $this->submitted(...);
	}

	/**
	 * Pass "sub"-form submission forward to custom submit function
	 *
	 * @param Form $form
	 *
	 * @return void
	 */
	public function submitted(\Nette\Forms\Form $form): void
	{
		if ($form->isSubmitted() instanceof SubmitButton === false)
			return;

		$btnName = $form->isSubmitted()->getName();
		if (!isset($form['groupActionCustom'][$btnName]))
			return;

		$values = $form->getValues();
		$values = $values['groupActionCustom'];

		$id = explode('_', $btnName)[1] ?? null;

		if ($id === null)
			return;

		$http_ids = $form->getHttpData(
			Form::DATA_LINE | Form::DATA_KEYS,
			strtolower($this->datagrid->getFullName()) . '_group_action_item[]',
		);
		$ids      = array_keys($http_ids);

		$this->groupActionsCustom[$id]->onSelect($ids, $values->{$id} ?? null);

		$form->getComponent('groupActionCustom')[$id]->setValue(null);
	}

	/**
	 * Add one group action (select box) to collection of actions
	 *
	 * @param string $title
	 * @param array  $options
	 *
	 * @return GroupAction
	 */
	public function addGroupSelectAction($title, $options): GroupAction
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupSelectAction($title, $options);
	}

	/**
	 * Add one group action (multiselect box) to collection of actions
	 *
	 * @param string $title
	 * @param array  $options
	 *
	 * @return GroupAction
	 */
	public function addGroupMultiSelectAction($title, $options): GroupAction
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupMultiSelectAction($title, $options);
	}

	/**
	 * Add one group action (text input) to collection of actions
	 *
	 * @param string $title
	 *
	 * @return GroupAction
	 */
	public function addGroupTextAction($title): GroupAction
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupTextAction($title);
	}

	/**
	 * Add one group action (textarea) to collection of actions
	 *
	 * @param string $title
	 *
	 * @return GroupAction
	 */
	public function addGroupTextareaAction($title): GroupAction
	{
		$id = ($s = sizeof($this->groupActionsCustom)) ? ($s + 1) : 1;

		return $this->groupActionsCustom[$id] = new GroupTextareaAction($title);
	}

	/**
	 * @param string $title
	 *
	 * @return GroupAction
	 */
	public function getGroupAction(string $title): GroupAction
	{
		foreach ($this->groupActionsCustom as $action) {
			if ($action->getTitle() === $title) {
				return $action;
			}
		}

		throw new DataGridGroupActionException("Group action $title does not exist.");
	}
}
