export class smapy {
    constructor(elementId, callback, initImmediately = true) {
        this.elementId = elementId
        this.callback = callback
        if (initImmediately) this.init()
    }
    init() {
        const initMap = () => {
            const map = new SMap(JAK.gel(this.elementId));
            map.addControl(new SMap.Control.Sync())
            map.addDefaultLayer(SMap.DEF_BASE).enable()

            const mouse = new SMap.Control.Mouse(SMap.MOUSE_PAN | SMap.MOUSE_WHEEL | SMap.MOUSE_ZOOM)
            map.addControl(mouse)

            map.addDefaultControls()

            this.callback(map)
        }

        const script = document.createElement("script");
        script.src = "https://api.mapy.cz/loader.js";
        script.onload = () => {
            window.Loader.async = true;
            window.Loader.load(
                null,
                null,
                initMap
            );
        };
        document.head.append(script);
    }

    /**
     * @param SMap map
     * @param Coord[] coords
     */
    static setCenterZoom(map, coords) {
        const cz = map.computeCenterZoom(coords)
        if (cz[0] && cz[1]) {
            map.setCenterZoom(cz[0], cz[1])
        }
    }
    static setZoom(map, ...params) {
        map.setZoom(...params)
    }
}

export class Marker {

    /**
     * @param SMap.Coords coord
     * @param string|null imgSrc
     * @param string|null title
     * @param Card|null card
     */
    static createInstance(coord, imgSrc = null, title = null, card = null) {
        const options = {
            anchor: {
                left: 10,
                bottom: 1
            }
        }

        if (imgSrc) {
            options.url = imgSrc
        }

        if (title) {
            options.title = title
        }

        const marker = new SMap.Marker(coord, null, options)

        if (card) {
            marker.decorate(SMap.Marker.Feature.Card, card)
        }

        return marker;
    }
}

export class Card {

    /**
     * @param string title -muze byt html
     * @param string body -muze byt html
     */
    static createInstance(title, body) {
        const card = new SMap.Card()
        card.getHeader().innerHTML = title
        card.getBody().innerHTML = body

        return card
    }

}

export class Coord {

    /**
     * @param float lat
     * @param float lng
     */
    static createInstance(lat, lng) {
        return SMap.Coords.fromWGS84(lng, lat)
    }

}

export class Layer {

    /**
     * @param SMap coords
     * @param Coord[] coords
     */
    static add(map, coords = []) {
        const layer = new SMap.Layer.Marker()
        map.addLayer(layer)
        layer.enable();
        for (let i = 0; i < coords.length; i++) {
            layer.addMarker(coords[i])
        }
    }

}

/**
 * @param SMap map
 * @param Object coords [{lat: 20.222, lng: 40.4005}]
 * @param string|null markerImgSrc
 * @param float|null zoom
 */
export function fastMapConfiguration(map, coords = [], markerImgSrc = null, zoom = null) {
    const markers = []
    const tmpCoords = []
    coords.forEach(el => {
        const coord = Coord.createInstance(el.lat, el.lng)
        markers.push(Marker.createInstance(coord, markerImgSrc))
        tmpCoords.push(coord)
    })

    Layer.add(map, markers)
    smapy.setCenterZoom(map, tmpCoords)

    if (zoom) {
        smapy.setZoom(map, zoom)
    }
}