<?php declare(strict_types = 1);

namespace Core\FrontModule\Model;

use Core\Model\Sites;
use Navigations\FrontModule\Model\LinksHelper;
use Nette\Application\LinkGenerator;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Providers\ISiteMapImage;
use Core\Model\Providers\ISiteMapUrl;
use Core\Model\Entities\EntityManagerDecorator;
use Navigations\Model\Navigations;
use Navigations\Model\Providers\INavigationItem;
use Nette\Http\Request;
use Nette\Utils\FileSystem;
use Thepixeldeveloper\Sitemap as DataSitemap;

/**
 * TODO cache - sledovat všechny entity které mají traitu TSeo a podle toho mazat cache
 */
class SiteMap
{
	const CHANGE_FREQ_ALWAYS  = 'always';
	const CHANGE_FREQ_HOURLY  = 'hourly';
	const CHANGE_FREQ_DAILY   = 'daily';
	const CHANGE_FREQ_WEEKLY  = 'weekly';
	const CHANGE_FREQ_MONTHLY = 'monthly';
	const CHANGE_FREQ_YEARLY  = 'yearly';
	const CHANGE_FREQ_NEVER   = 'never';

	const TMP_DIR = TMP_DIR . '/sitemap';

	public static string $siteTmpDir = self::TMP_DIR;

	protected EntityManagerDecorator $em;
	protected LinkGenerator          $linkGenerator;
	protected Request                $httpRequest;
	protected Navigations            $navigations;
	protected Sites                  $sites;
	protected LinksHelper            $linksHelper;

	protected array $componentChildC = [];

	public function __construct(
		EntityManagerDecorator $em,
		LinkGenerator          $linkGenerator,
		Request                $request,
		Navigations            $navigations,
		Sites                  $sites,
		LinksHelper            $linksHelper
	)
	{
		$this->em            = $em;
		$this->linkGenerator = $linkGenerator;
		$this->httpRequest   = $request;
		$this->navigations   = $navigations;
		$this->sites         = $sites;
		$this->linksHelper   = $linksHelper;
	}

	/**
	 * Prohledá navigaci a její komponenty. Sestaví Xml a vrátí
	 *
	 * @param string $group
	 * @param int    $page
	 *
	 * @return DataSitemap\SitemapIndex|DataSitemap\Urlset|null
	 * @throws \Nette\Application\UI\InvalidLinkException
	 */
	public function getSiteMap(string $lang, $group = 'default', $page = 0)
	{
		self::$siteTmpDir = self::TMP_DIR . '/' . $this->sites->getCurrentSite()->getIdent() . '-' . $lang;
		$oGroup           = $group;
		/** @var DaoNavigationItem[] $navigations */
		$navigations = $this->navigations->getPublishedToDaoNavigationItem($lang);

		$pageLimit = 49500;

		$site          = $this->sites->getCurrentSite();
		$currentDomain = $site->getDomains()[$lang] ?? $site->getCurrentDomain();

		// Pár základních proměnných
		$domain = rtrim($this->httpRequest->getUrl()->scheme . '://' . $currentDomain->getDomain(), '/') . '/';
		/** @var DataSitemap\Sitemap[] $urls */
		$urls      = [];
		$urlsChunk = [];
		$totalUrl  = 0;
		$ci        = 1;
		$files     = [];
		FileSystem::createDir(self::$siteTmpDir . '/' . $oGroup . '/');

		foreach (glob(self::$siteTmpDir . '/' . $oGroup . '/*') ?: [] as $k => $f) {
			$files[$k + 1] = $f;
		}

		if (empty($files)) {
			foreach ($navigations as $nav) {
				if (!$this->canAddToSiteMap($nav)) {
					continue;
				}

				$params = ['activeNavigation' => $nav];

				if (!$nav->link) {
					continue;
				}

				$link = $domain . ltrim(is_array($nav->link) ? $nav->link[0] : $nav->link, '/');
				// TODO obrázky
				$urls[$link] = $this->createUrl($link, $nav);

				// Vyhledání komponenty a vytvoření URL pokud obsahuje nějaké položky
				$component = $this->navigations->itemsCollector->getItemById($nav->componentType);
				$clearI    = 0;
				if ($component) {
					if (property_exists($component, 'linkGenerator')) {
						$component->linkGenerator = $this->linkGenerator;
					}

					foreach ($this->getComponentChild($lang, $component, $nav, true) as $v) {
						if (isset($v['activeNavigation'])) {
							unset($v['activeNavigation']);
						}

						if (!isset($v['locale'])) {
							$v['locale'] = $lang;
						}

						if (isset($v['siteMapData'][$group]) && $this->canAddToSiteMap($v['siteMapData'][$group])) {
							if (isset($v['link'])) {
								$link = $v['link'];
							} else {
								$link = $domain . ltrim(($v['linkPath'] ?: $this->navigations->getUrlById($v)), '/');
							}

							$url = $this->createUrl($link, $v['siteMapData'][$group]);

							$tmp = $v;
							unset($tmp['siteMapData']);
							foreach ($this->linksHelper->getOtherLangLinks($nav, $tmp, $v['presenter']) as $linkLang => $otherLangLink) {
								if ($linkLang == $lang) {
									continue;
								}

								$linkExt = new DataSitemap\Extensions\Link($linkLang, $otherLangLink['link']);
								$url->addExtension($linkExt);
							}

							foreach ($v['siteMapData'][$group]->getImages() as $img) {
								$url->addExtension($this->createImage($domain . ltrim((string) $img->getFilePath(), '/'), $img));
							}
							$urls[$link] = $url;
							$totalUrl++;

							if (count($urls) > $pageLimit) {
								$urlsChunk[$ci] = $urls;
								$urls           = [];
								$ci++;
							}
						}
						$clearI++;
						if ($clearI % 1500 == 0) {
							$this->em->clear();
						}

						if ($clearI % 3000 == 0) {
							gc_collect_cycles();
						}
					}
				}
			}

			if (!empty($urls)) {
				$urlsChunk[$ci] = $urls;
				$urls           = [];
			}
		} else {
			$totalUrl = (int) file_get_contents(self::$siteTmpDir . "/$oGroup/count.txt");
		}

		if ($totalUrl > $pageLimit && $page == 0) {
			$urlSet = new DataSitemap\SitemapIndex();
			foreach ($urlsChunk as $fp => $url) {
				$chunkLink = $this->httpRequest->getUrl()->absoluteUrl . "sitemap-$oGroup-$lang-$fp.xml";
				$chunkUrl  = new DataSitemap\Sitemap($chunkLink);
				//TODO last MOD

				$urlSet->add($chunkUrl);
			}
		} else {
			$urlSet = new DataSitemap\Urlset();
			/** @var DataSitemap\Sitemap[] $urls */
			$urls = $urlsChunk[$page > 0 ? $page : 1];

			if (empty($urls)) {
				return null;
			}

			foreach ($urls as $v) {
				$urlSet->add($v);
			}
		}

		return $urlSet;
	}

	/**
	 * @param string      $url
	 * @param ISiteMapUrl $data
	 *
	 * @return DataSitemap\Url
	 */
	protected function createUrl(string $url, $data)
	{
		$url = new DataSitemap\Url($url);
		if ($data->getModified()) {
			$url->setLastMod($data->getModified());
		}

		if ($data->getSeo('siteMapChangeFreq')) {
			$url->setChangeFreq((string) $data->getSeo('siteMapChangeFreq'));
		}

		if ($data->getSeo('siteMapPriority')) {
			$url->setPriority(str_replace(',', '.', (string) $data->getSeo('siteMapPriority')));
		}

		return $url;
	}

	/**
	 * @param string        $url
	 * @param ISiteMapImage $img
	 *
	 * @return DataSitemap\Extensions\Image
	 */
	protected function createImage(string $url, ISiteMapImage $img)
	{
		$image = (new DataSitemap\Extensions\Image($url))
			->setCaption($img->getCaption())
			->setGeoLocation($img->getGeoLocation())
			->setTitle($img->getTitle())
			->setLicense($img->getLicense());

		return $image;
	}

	/**
	 * @param ISiteMapUrl $data
	 *
	 * @return bool
	 */
	protected function canAddToSiteMap($data)
	{
		if (method_exists($data, 'getSeo') &&
			($data->getSeo('addToSiteMap') !== '1' || strpos((string) $data->getSeo('robots'), 'noindex') !== false)
		) {
			return false;
		}

		return true;
	}

	/**
	 * @return array
	 * @throws \ReflectionException
	 */
	public function getAllChangeFreq(): array
	{
		$refl = new \ReflectionClass(self::class);
		$arr  = [];

		foreach ($refl->getConstants() as $k => $v) {
			if (strpos($k, 'CHANGE_FREQ_') === 0) {
				$arr[$v] = $v;
			}
		}

		return $arr;
	}

	/**
	 * @param INavigationItem   $component
	 * @param DaoNavigationItem $nav
	 * @param bool              $sitemapData
	 *
	 * @return mixed
	 */
	public function getComponentChild(string $lang, INavigationItem $component, $nav, $sitemapData = true)
	{
		$key = $nav->getId() . '-' . $lang;
		$i   = 0;
		if (!isset($this->componentChildC[$key])) {
			$data = [];
			do {
				$tmp = $component->getComponentChild($lang, $nav, $sitemapData);
				foreach ($tmp as $v)
					$data[] = $v;
				$i++;
			} while (isset($tmp['continue']) && $tmp['continue'] === true && $i < 500);

			$this->componentChildC[$key] = $data;
		}

		return $this->componentChildC[$key];
	}
}
