<?php declare(strict_types = 1);

namespace Core\Model;

use Core\Model\Entities\Country;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Dao;
use Core\Model\Helpers\Strings;
use Nette\Caching\Cache;
use Contributte\Translation\Translator;

/**
 * @method Country|null getReference($id)
 */
class Countries extends BaseEntityService
{
	public const CACHE_NAMESPACE = 'countries';

	/** @var string */
	protected $entityClass = Country::class;

	protected Translator $translator;

	protected ?array $cOptionsForSelect = null;

	/** @var null|Dao\Country[] */
	protected ?array $cDao = null;
	protected ?array $cAll = null;

	public function __construct(
		Translator $translator
	)
	{
		$this->translator = $translator;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @param string $id
	 */
	public function get($id): ?Country
	{
		$countries = [];
		foreach ($this->getAll() as $country) {
			$countries[strtolower($country->getId())] = $country;
			$countries[strtoupper($country->getId())] = $country;
		}

		return $countries[$id] ?? null;
	}

	public function checkId(string $id): ?string
	{
		foreach ($this->getDao() as $countryId => $country) {
			if (strtolower((string) $id) === $countryId || strtoupper((string) $id) === $countryId) {
				return $countryId;
			}
		}

		return null;
	}

	public function setPosition(string $id, int $position): bool
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/** @return Country[]|null */
	public function getAll()
	{
		if ($this->cAll === null) {
			$countriesQuery = $this->getEr()->createQueryBuilder('c', 'c.id');
			$countriesQuery->orderBy('c.position');

			$this->cAll = $countriesQuery->getQuery()->getResult();
		}

		return $this->cAll;
	}

	/**
	 * @return Dao\Country[]
	 */
	public function getDao(): array
	{
		if ($this->cDao === null) {
			$this->cDao = $this->getCache()->load(self::CACHE_NAMESPACE, function(&$dep) {
				$dep = [
					Cache::EXPIRATION => '1 month',
				];

				$data = [];
				foreach ($this->getEr()->createQueryBuilder('c')
					         ->orderBy('c.position')->getQuery()
					         ->getArrayResult() as $row) {
					$data[$row['id']] = new Dao\Country(
						$row['id'],
						$row['iso3166_1'],
						$row['name'],
					);
				}

				return $data;
			});

			foreach ($this->cDao as $dao) {
				$name = $this->translator->translate('country.' . Strings::lower($dao->getId()));

				if ($name) {
					$dao->name = $name;
				}
			}
		}

		return $this->cDao;
	}

	public function getAllNameColumn(): array
	{
		if ($this->cOptionsForSelect === null) {
			$this->cOptionsForSelect = [];
			foreach ($this->getEr()->createQueryBuilder('c')->select('c.id, c.name')
				         ->orderBy('c.position')->getQuery()->getScalarResult() as $row) {
				$this->cOptionsForSelect[$row['id']] = $this->translator->translate('country.' . Strings::lower($row['id']));
			}
		}

		return $this->cOptionsForSelect;
	}
}
