<?php declare(strict_types = 1);

namespace Core\Model\Helpers;

use Core\Model\SystemConfig;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use \Nette\Utils\Strings as NetteStrings;

class Strings extends NetteStrings
{
	public static bool $priceKeepTail = false;

	protected static array $cFormatted = [];

	/**
	 * @param float       $price
	 * @param string|null $currency
	 * @param string|null $separator
	 * @param bool        $currencyFromLeft
	 * @param string|null $decSep
	 * @param int|null    $decimals
	 *
	 * @return string
	 */
	public static function priceFormat(float $price, ?string $currency = 'Kč', ?string $separator = ' ', bool $currencyFromLeft = false, ?string $decSep = ',', ?int $decimals = 2): string
	{
		$decPoint       = $decSep;
		$priceFormatted = number_format($price, $decimals, $decPoint, ' ');

		// e.g.: ,00
		$tail = $decPoint . implode('', array_fill(0, $decimals, 0) ?: []);

		if (!self::$priceKeepTail && NetteStrings::endsWith($priceFormatted, $tail)) {
			$priceFormatted = substr($priceFormatted, 0, strlen($priceFormatted) - strlen($tail));
		}

		return sprintf(
			'%s%s%s',
			$currencyFromLeft ? $currency : $priceFormatted,
			$separator ?? ' ',
			$currencyFromLeft ? $priceFormatted : $currency
		);
	}

	/**
	 * @param string      $value
	 * @param string|null $valueRegion
	 * @param bool        $showRegion
	 *
	 * @return string
	 */
	public static function phoneFormat(string $value, ?string $valueRegion = null, bool $showRegion = false): string
	{
		$key = $value . '_' . ($valueRegion ?: '') . '_' . ($showRegion ? '1' : '0');
		if (isset(self::$cFormatted[$key])) {
			return self::$cFormatted[$key];
		}

		try {
			$phoneUtils  = PhoneNumberUtil::getInstance();
			$valid       = false;
			$phone       = null;
			$region      = null;
			$plusInValue = false;

			if ($valueRegion) {
				$regions = [$valueRegion] + $phoneUtils->getSupportedRegions();
			} else {
				$regions = SystemConfig::load('phoneInputRegions') + $phoneUtils->getSupportedRegions();
			}

			foreach ($regions as $region) {
				if (substr($value, 0, 1) === '+') {
					$phone       = $phoneUtils->parse($value);
					$plusInValue = true;
				} else {
					$phone = $phoneUtils->parse($value, strtoupper($region));
				}

				$valid = $phoneUtils->isValidNumber($phone);

				if ($valid) {
					$region = $phoneUtils->getRegionCodeForCountryCode($phone->getCountryCode());
					break;
				} else if ($plusInValue) {
					break;
				}
			}

			self::$cFormatted[$key] = $valid && $phone
				? $phoneUtils->format($phone, PhoneNumberFormat::INTERNATIONAL) . ($showRegion ? ' (' . $region . ')' : '')
				: $value;

			return self::$cFormatted[$key];
		} catch (\Exception $e) {
		}

		return $value;
	}

	public static function joinPaths(string ...$paths): string
	{
		return preg_replace('~[/\\\\]+~', DIRECTORY_SEPARATOR, implode(DIRECTORY_SEPARATOR, $paths));
	}

	public static function isValidTimestamp(int $timestamp): bool
	{
		return ($timestamp <= PHP_INT_MAX) && ($timestamp >= ~PHP_INT_MAX);
	}

	/** @param float|int|string $value */
	public static function formatEntityDecimal($value): ?string
	{
		$res = str_replace(',', '.', (string) $value);

		return $res === '' ? null : $res;
	}

	public static function normalizeText(string $value): string
	{
		$value = strip_tags($value);
		$value = str_replace('&nbsp;', ' ', $value);
		$value = preg_replace('/\s\s+/', ' ', $value);

		return $value;
	}

	public static function wordsCount(string $value): int
	{
		return count(explode(' ', $value));
	}

	public static function removeFromStart(string $origin, string $remove): string
	{
		if (static::startsWith($origin, $remove))
			return trim(static::substring($origin, strlen($remove)));

		return $origin;
	}

	public static function toUtf8(string $value): string
	{
		// detect UTF-8
		if (preg_match('#[\x80-\x{1FF}\x{2000}-\x{3FFF}]#u', $value)) {
			return $value;
		}

		// detect WINDOWS-1250
		if (preg_match('#[\x7F-\x9F\xBC]#', $value)) {
			return (string) iconv('WINDOWS-1250', 'UTF-8', $value);
		}

		// assume ISO-8859-2
		return (string) iconv('ISO-8859-2', 'UTF-8', $value);
	}

	public static function toCamelCase(string $value): string
	{
		$value = self::capitalize($value);
		$value = self::webalize($value, null, false);
		$value = str_replace('-', '', $value);
		$value = lcfirst($value);

		return $value;
	}

	public static function sortCzech(string $a, string $b): int
	{
		$czechCharsS = ['Á', 'Č', 'Ď', 'É', 'Ě', 'Ch', 'Í', 'Ň', 'Ó', 'Ř', 'Š', 'Ť', 'Ú', 'Ů', 'Ý', 'Ž', 'á', 'č', 'ď',
			'é', 'ě', 'ch', 'í', 'ň', 'ó', 'ř', 'š', 'ť', 'ú', 'ů', 'ý', 'ž'];
		$czechCharsR = ['AZ', 'CZ', 'DZ', 'EZ', 'EZZ', 'HZZZ', 'IZ', 'NZ', 'OZ', 'RZ', 'SZ', 'TZ', 'UZ', 'UZZ', 'YZ',
			'ZZ', 'az', 'cz', 'dz', 'ez', 'ezz', 'hzzz', 'iz', 'nz', 'oz', 'rz', 'sz', 'tz', 'uz', 'uzz', 'yz', 'zz'];

		$a = str_replace($czechCharsS, $czechCharsR, $a);
		$b = str_replace($czechCharsS, $czechCharsR, $b);

		return strnatcasecmp($a, $b);
	}

	public static function fixHtmlString(string $html): string
	{
		$html = preg_replace('/(?:^<p>&nbsp;<\/p>)|(?:<p>&nbsp;<\/p>$)/', '', $html);
		$html = str_replace(["\r", "\n"], '', $html);
		$html = trim($html);

		if (self::startsWith($html, '<p><p>') && self::endsWith($html, '</p></p>')) {
			$html = str_replace(['<p><p>', '</p></p>'], ['<p>', '</p>'], $html);
		}

		return $html;
	}
}
