<?php declare(strict_types = 1);

namespace Core\Model\Images\Macros;

use Core\Model\Images\ImagePipe;
use Core\Model\SystemConfig;
use Latte;
use Latte\CompileException;
use Latte\Compiler;
use Latte\MacroNode;
use Latte\Macros\MacroSet;
use Latte\PhpWriter;
use Nette;

class Macros extends MacroSet
{
	private bool    $isUsed = false;
	protected array $configParams;

	public function __construct(Latte\Compiler $compiler, array $params = [])
	{
		parent::__construct($compiler);
		$this->configParams = $params;
	}

	public static function install(Compiler $compiler, array $params): self
	{
		$me = new self($compiler, $params);

		/**
		 * {img [namespace/]$name[, $size[, $flags]]}
		 */
		$me->addMacro('img', [$me, 'macroImg'], null, [$me, 'macroAttrImg']);
		$me->addMacro('imgLazy', [$me, 'macroImgLazy'], null, [$me, 'macroAttrImgLazy']);

		return $me;
	}

	public function macroImg(MacroNode $node, PhpWriter $writer): string
	{
		$data = $this->prepareMacroImg($node, $writer);

		$return = $writer->write('echo %escape(' . $writer->formatWord($data['src']) . ')');

		if (SystemConfig::load('images.autoGeneratedSize', false) && isset($data['width'], $data['height']) && $node->htmlNode && $node->htmlNode->name === 'img') {
			$return .= '."\" width=\"' . $writer->formatWord((string) $data['width']) . '\" height=\"' . $writer->formatWord((string) $data['height']) . '"';
		}

		return $return;
	}

	public function macroAttrImg(MacroNode $node, PhpWriter $writer): string
	{
		$data = $this->prepareMacroAttrImg($node, $writer);

		$return = '?>';
		$return .= $writer->write(' src="<?php echo %escape(' . $writer->formatWord($data['src']) . ')?>"');

		if ($data['srcset']) {
			$return .= $writer->write(' srcset="<?php echo %escape(' . $writer->formatWord($data['srcset']) . ')?>"');
		}

		if (SystemConfig::load('images.autoGeneratedSize', false) && isset($data['width'], $data['height'])) {
			$return .= $writer->write(' width="<?php echo %escape(' . $writer->formatWord((string) $data['width']) . ')?>"');
			$return .= $writer->write(' height="<?php echo %escape(' . $writer->formatWord((string) $data['height']) . ')?>"');
		}

		$return .= '<?php';

		return $return;
	}

	public function macroImgLazy(MacroNode $node, PhpWriter $writer): string
	{
		$data = $this->prepareMacroImg($node, $writer);

		if (SystemConfig::load('images.autoGeneratedSize', false) && isset($data['width'], $data['height'])) {
			$writer->write('?> width="<?php echo (' . $writer->formatWord((string) $data['width']) . ') ?>"');
			$writer->write('?> height="<?php echo (' . $writer->formatWord((string) $data['height']) . ') ?>"');
		}

		return $this->configParams['lazyLoading'] === true
			? $writer->write('?> src="<?php echo $_imagePipe->generatePlaceholder(' . ($data['placeholderWidth'] ?? 'null') . ', ' . ($data['placeholderHeight'] ?? 'null') . '); ?>" data-src="<?php echo (' . $writer->formatWord($data['src']) . ') ?>"<?php')
			: $writer->write('?> src="<?php echo %escape(' . $writer->formatWord($data['src']) . ') ?>"<?php');
	}

	public function macroAttrImgLazy(MacroNode $node, PhpWriter $writer): string
	{
		if ($this->configParams['lazyLoading'] !== true)
			return $this->macroAttrImg($node, $writer);

		$data = $this->prepareMacroAttrImg($node, $writer);

		$return = '?>';
		$return .= $writer->write(' src="<?php echo $_imagePipe->generatePlaceholder(' . ($data['placeholderWidth'] ?? 'null') . ', ' . ($data['placeholderHeight'] ?? 'null') . '); ?>" data-src="<?php echo %escape(' . $writer->formatWord($data['src']) . ')?>"');

		if ($data['srcset']) {
			$return .= $writer->write(' data-srcset="<?php echo %escape(' . $writer->formatWord($data['srcset']) . ')?>"');
		}

		if (SystemConfig::load('images.autoGeneratedSize', false) && isset($data['width'], $data['height'])) {
			$return .= $writer->write(' width="<?php echo %escape(' . $writer->formatWord((string) $data['width']) . ')?>"');
			$return .= $writer->write(' height="<?php echo %escape(' . $writer->formatWord((string) $data['height']) . ')?>"');
		}

		$return .= '<?php';

		return $return;
	}

	protected function prepareMacroImg(MacroNode $node, PhpWriter $writer): array
	{
		$this->isUsed = true;
		$arguments    = Helpers::prepareMacroArguments($node->args);
		if ($arguments["name"] === null) {
			throw new CompileException("Please provide filename.");
		}

		$namespace = $arguments["namespace"];
		unset($arguments["namespace"]);
		$arguments = array_map(function($value) use ($writer) {
			return $value ? $writer->formatWord($value) : 'NULL';
		}, $arguments);

		$command = '$_imagePipe';
		$command .= $namespace !== null ? '->setNamespace(' . $writer->formatWord(trim($namespace)) . ')' : '';
		$command .= '->request(' . implode(", ", $arguments) . ')';

		$return = [
			'src' => $command,
		];

		if (isset($arguments['size']) && Nette\Utils\Strings::contains($arguments['size'], 'x')) {
			[$w, $h] = explode('x', str_replace(["'", '"'], '', $arguments['size']));
			$return['placeholderWidth']  = $w;
			$return['placeholderHeight'] = $h;

			if (SystemConfig::load('images.autoGeneratedSize', false)) {
				$return['width']  = $w;
				$return['height'] = $h;
			}
		}

		return $return;
	}

	protected function prepareMacroAttrImg(MacroNode $node, PhpWriter $writer): array
	{
		$this->isUsed = true;
		$arguments    = Helpers::prepareMacroArguments($node->args);
		if ($arguments["name"] === null) {
			throw new CompileException("Please provide filename.");
		}

		$namespace = $arguments["namespace"];
		unset($arguments["namespace"]);
		[$width, $height] = isset($arguments['size'])
			? array_map('intval', explode('x', $arguments['size'])) : [0, 0];
		$arguments = array_map(function($value) use ($writer) {
			return $value ? $writer->formatWord($value) : 'NULL';
		}, $arguments);

		$srcset      = [];
		$srcsetSizes = $this->configParams['srcsetSizes'];
		foreach ($srcsetSizes as $k => $sizes) {
			$newWidth = $sizes['w'];

			if (isset($srcsetSizes[$k - 1]) && $srcsetSizes[$k - 1]['w'] > $width) {
				break;
			}

			$tmp         = $arguments;
			$newSize     = Helpers::calculateNewSize($width, $height, $newWidth);
			$tmp['size'] = $writer->formatWord("{$newSize['width']}x{$newSize['height']}");
			$srcset[]    = '$_imagePipe->request(' . implode(", ", $tmp) . ') . " ' . $newWidth . 'w"';
		}

		$srcset  = implode('.",".', $srcset);
		$command = '$_imagePipe->request(' . implode(", ", $arguments) . ')';

		$return = [
			'srcset'            => $srcset,
			'src'               => $command,
			'placeholderWidth'  => $width,
			'placeholderHeight' => $height,
		];

		if (SystemConfig::load('images.autoGeneratedSize', false)) {
			$return['width']  = $width;
			$return['height'] = $height;
		}

		return $return;
	}

	public function initialize(): void { $this->isUsed = false; }

	public function finalize(): array
	{
		if (!$this->isUsed) {
			return [];
		}

		return [
			get_called_class() . '::validateTemplateParams($template);',
			null,
		];
	}

	/**
	 * @param \Latte\Runtime\Template|Nette\Application\UI\Template $template
	 */
	public static function validateTemplateParams($template): void
	{
		if (!method_exists($template, 'getParameters')) {
			return;
		}

		$params = $template->getParameters();
		if (!isset($params['_imagePipe']) || !$params['_imagePipe'] instanceof ImagePipe) {
			$where = isset($params['control']) ?
				" of component " . get_class($params['control']) . '(' . $params['control']->getName() . ')'
				: null;

			throw new Nette\InvalidStateException(
				'Please provide an instanceof Img\\ImagePipe ' .
				'as a parameter $_imagePipe to template' . $where
			);
		}
	}

	public function setParams(array $params): void { }
}
