<?php declare(strict_types = 1);

namespace Core\Model\Templating\Filters;

use Core\Model\Helpers\CssHelper;
use Core\Model\Images\ImagePipe;
use Core\Model\Images\Macros\Helpers;
use Core\Model\SystemConfig;
use Contributte\Translation\Translator;
use Nette\Utils\Image;
use Nette\Utils\Strings;
use PhpOffice\PhpSpreadsheet\Reader\Html;
use voku\helper\HtmlDomParser;

class EditorData
{
	protected Translator $translator;
	protected ImagePipe  $imagePipe;

	public function __construct(Translator $translator, ImagePipe $imagePipe)
	{
		$this->translator = $translator;
		$this->imagePipe  = $imagePipe;
	}

	/**
	 * @param string|null $string
	 *
	 * @return string
	 */
	public function __invoke(?string $string): ?string
	{
		if (!$string)
			return $string;

		$html = HtmlDomParser::str_get_html($string);

		$readMore = $html->find('.ck-readmore');

		$mText = $this->translator->translate('front.editor.readMore');
		$lText = $this->translator->translate('front.editor.readLess');
		foreach ($readMore as $k => $rm) {
			$prev   = $rm->prev_sibling();
			$method = $prev->tag != 'p' ? 'outertext' : 'innertext';

			$prev->$method .= ' <a href="#readMore' . $k . '" class="ck-readmore-trigger" data-more="' . $mText . '" data-less="' . $lText . '">' . $mText . '</a>';
			$rm->id        = 'readMore' . $k;
		}

		// original image replace by image preview
		$document = new \DOMDocument;
		$document->loadHTML(mb_convert_encoding($string, 'HTML-ENTITIES', 'UTF-8'));
		$lazyLoading = SystemConfig::load('images.lazyLoading', false);
		$srcsetSizes = SystemConfig::load('images.srcsetSizes', []);
		foreach ($document->getElementsByTagName('img') as $el) {
			$src   = trim(str_replace('%20', ' ', $el->getAttribute('src')));
			$sizes = CssHelper::parse($el->getAttribute('style'));

			foreach (['width', 'height'] as $s) {
				if (isset($sizes[$s])) {
					$sizes[$s] = (int) str_replace('px', '', (string) $sizes[$s]);
				} else {
					$sizes[$s] = 0;
				}
			}

			$width  = $sizes['width'];
			$height = $sizes['height'];

			if ((!$width || !$height) && Strings::startsWith($src, '/')) {
				$tmp = @getimagesize(WWW_DIR . $src);
				if ($tmp) {
					[$width, $height] = $tmp;
				}
			}

			$srcset = [];
			foreach ($srcsetSizes as $k => $sizes) {
				$newWidth = $sizes['w'];

				if (!$width || !$height || isset($srcsetSizes[$k - 1]) && $srcsetSizes[$k - 1]['w'] > $width)
					break;

				$newSize  = Helpers::calculateNewSize($width, $height, $newWidth);
				$srcset[] = $this->imagePipe->request($src, "{$newSize['width']}x{$newSize['height']}", Image::FIT) . ' ' . $newWidth . 'w';
			}

			$src = $this->imagePipe->request($src, "{$width}x{$height}", Image::FIT);
			if (!$width) {
				$imageSize = @getimagesize($src);

				if (isset($imageSize[0]) && $imageSize[0] > 0) {
					$el->setAttribute('style', $el->getAttribute('style') . ';width:' . $imageSize[0] . 'px');
				}
			}

			if ($lazyLoading) {
				$el->removeAttribute('src');
				$el->setAttribute('data-src', $src);
				if ($srcset) {
					$el->setAttribute('data-srcset', implode(',', $srcset));
				}
			} else {
				$el->setAttribute('src', $src);
				if ($srcset) {
					$el->setAttribute('srcset', implode(',', $srcset));
				}
			}
		}

		return (string) $document->saveHTML();
	}
}
