<?php declare(strict_types = 1);

namespace Core\Model\UI;

use Contributte\Application\UI\BasePresenter;
use Core\Components\Header\Favicon;
use Core\Components\Header\Header;
use Core\Components\Header\Meta;
use Core\Components\Header\Title;
use Core\Components\IPartRenderControlFactory;
use Core\Components\JsTranslates\IJsTranslatesFactory;
use Core\Components\JsTranslates\JsTranslates;
use Core\Components\PartRenderControl;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\EventDispatcher;
use Core\AdminModule\Components\ContentLangSwitcher\ContentLangSwitcher;
use Core\AdminModule\Components\ContentLangSwitcher\IContentLangSwitcherFactory;
use Core\Components\Flashes\Flashes;
use Core\Components\Flashes\IFlashesFactory;
use Core\Components\Header\IFaviconFactory;
use Core\Components\Header\IHeaderFactory;
use Core\Components\Header\IMetaFactory;
use Core\Components\Header\ITitleFactory;
use Core\FrontModule\Components\ISearchFormFactory;
use Core\FrontModule\Components\SearchForm;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\DefaultLang;
use Core\Model\Lang\Lang;
use Core\Model\Lang\Langs;
use Core\Model\Module;
use Core\Model\Settings;
use Core\Model\SystemConfig;
use Forms\FrontModule\Components\FormControl;
use Forms\FrontModule\Components\IFormControlFactory;
use Kdyby\Autowired\AutowireComponentFactories;
use Navigations\FrontModule\Components\ILangSwitcherFactory;
use Navigations\FrontModule\Components\LangSwitcher;
use Nette\Application\UI\ComponentReflection;
use Nette\Application\UI\Multiplier;
use Contributte\Translation\Translator;
use Symfony\Contracts\EventDispatcher\Event;

abstract class AbstractPresenter extends BasePresenter
{
	use AutowireComponentFactories;
	use TUITemplate;

	/** @persistent */
	public ?string $locale = null;

	/** @var Translator @inject */
	public $translator;

	/** @var ImagePipe @inject */
	public $imagePipe;

	/** @var Langs @inject */
	public $langsService;

	/** @var SystemConfig @inject */
	public $systemConfig;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var Settings|null @inject */
	public ?Settings $settings;

	/** @var Lang */
	protected $lang;

	public string $baseUrl = '';

	protected array $ajaxPayloadPlaceholders = [];

	/**
	 * @param ComponentReflection $element
	 */
	public function checkRequirements($element): void
	{
		Module::$currentPresenterName = $this->getFullModuleName();
		parent::checkRequirements($element);
	}

	protected function startup()
	{
		parent::startup();
		$this->eventDispatcher->dispatch(new ControlEvent($this), self::class . '::startup');
		$this->dispatchStaticEvent(new ControlEvent($this), 'startup');


		foreach ($this->translator->getLocalesWhitelist() as $l) {
			if ($this->locale && strpos($l, $this->locale) === 0) {
				SetLocale(LC_ALL, $l . ".utf8");
				break;
			}
		}

		$this->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();
		$this->lang    = $this->langsService->getLang($this->translator->getLocale());
	}

	public function beforeRender()
	{
		parent::beforeRender();
		$this->template->title          = $this['title']->getTitle();
		$this->template->fullModuleName = $this->getFullModuleName();
		$this->template->moduleName     = $this->getModuleName();
		$this->template->presenterName  = $this->getName();

		$reflect                        = new \ReflectionClass($this);
		$this->template->originTemplate = dirname((string) $reflect->getFileName()) . '/../Templates/' . str_replace('Presenter.php', '', basename((string) $reflect->getFileName())) . '/' . lcfirst((string) $this->getAction()) . '.latte';

		$this->eventDispatcher->dispatch(new ControlEvent($this), self::class . '::beforeRender');
		$this->dispatchStaticEvent(new ControlEvent($this), 'beforeRender');

		// SEO stránkování
		foreach ($this->getComponents() as $component) {
			if (method_exists($component, 'getPaginator')) {
				$page = $component->getPaginator()->getPage();
				if ($page > 1) {
					$link = $this->link('this', [
						$component->getName() . '-page' => $page - 1,
						'do'                            => $component->getName() . '-paginator',
					]);
					$this['meta']->setOtherHeader('<link rel="prev" href="' . $link . '">');
				}
				if ($page < $component->getPaginator()->getPageCount()) {
					$link = $this->link('this', [
						$component->getName() . '-page' => $page + 1,
						'do'                            => $component->getName() . '-paginator',
					]);
					$this['meta']->setOtherHeader('<link rel="next" href="' . $link . '">');
				}
			}
		}
	}

	protected function tryCall(string $method, array $params): bool
	{
		$result = parent::tryCall($method, $params);

		if ($result) {
			$this->dispatchStaticEvent(new ControlEvent($this), $method);
		}

		return $result;
	}

	/**
	 * @param string|null $colonText
	 */
	protected function setTitle(string $title, $colonText = null): void
	{
		$this['title']->setTitle($title, false);
		$this['title']->setColonText($colonText);
	}

	/**
	 * @param string|array $title
	 */
	protected function setHeader($title = null, ?string $icon = null, ?string $description = null): void
	{
		if (is_string($title)) {
			$this['header']->setTitle($this->translator->translate($title));
			$this['title']->setTitle($this->translator->translate($title), false);
		} else if (is_array($title)) {
			$this['header']->setTitle($this->translator->translate(...$title));
			$this['title']->setTitle($this->translator->translate(...$title), false);
		} else {
			$this['header']->setTitle($this['title']->getTitle());
		}

		if (!empty($icon)) {
			$this['header']->setIcon($icon);
		} else {
			$this['header']->setIcon('fas fa-table');
		}

		if (!empty($description)) {
			$this['header']->setDescription($this->translator->translate($description));
		}
	}

	/*******************************************************************************************************************
	 *=========================== Components
	 */

	protected function createComponentTitle(ITitleFactory $factory): Title { return $factory->create(); }

	protected function createComponentFlashes(IFlashesFactory $factory): Flashes { return $factory->create(); }

	protected function createComponentMeta(IMetaFactory $factory): Meta { return $factory->create(); }

	protected function createComponentFavicon(IFaviconFactory $factory): Favicon { return $factory->create(); }

	protected function createComponentHeader(IHeaderFactory $factory): Header { return $factory->create(); }

	protected function createComponentSearchForm(ISearchFormFactory $factory): SearchForm { return $factory->create(); }

	protected function createComponentLangSwitcher(ILangSwitcherFactory $factory): LangSwitcher { return $factory->create(); }

	protected function createComponentContentLangSwitcher(IContentLangSwitcherFactory $factory): ContentLangSwitcher { return $factory->create(); }

	protected function createComponentPartControl(IPartRenderControlFactory $factory): PartRenderControl { return $factory->create(); }

	protected function createComponentFormControl(IFormControlFactory $factory): Multiplier
	{
		return new Multiplier(function(string $formId) use ($factory): FormControl {
			return $factory->create((int) $formId);
		});
	}

	protected function createComponentJsTranslates(IJsTranslatesFactory $factory): JsTranslates { return $factory->create(); }

	/*******************************************************************************************************************
	 *=========================== Helpers
	 */

	protected function dispatchStaticEvent(Event $event, string $eventName): void
	{
		$class = static::class;

		if (strpos($class, 'OverridePresenter') !== false) {
			$this->eventDispatcher->dispatch($event, $class . '::' . $eventName);
			$class = str_replace('OverridePresenter', 'Presenter', $class);
		}

		$this->eventDispatcher->dispatch($event, $class . '::' . $eventName);
	}

	public function getFullModuleName(): string
	{
		$arr = explode(':', $this->name);
		array_pop($arr);

		return implode(':', $arr);
	}

	/**
	 * @param mixed          $message
	 * @param int|array|null $count
	 */
	public function t($message, $count = null, array $parameters = [], ?string $domain = null, ?string $locale = null): string
	{
		return $this->translator->translate($message, $count, $parameters, $domain, $locale);
	}

	/**
	 * @param string|array $value
	 */
	public function setPayloadAjaxPlaceholder(string $key, $value): void
	{
		$this->ajaxPayloadPlaceholders[$key] = $value;
	}

	public function sendPayload(): void
	{
		$this->payload->ajaxPlaceholders = $this->ajaxPayloadPlaceholders;
		parent::sendPayload();
	}

	/*******************************************************************************************************************
	 *=========================== Flashes
	 */

	public function flashMessageSuccess(string $message): \stdClass
	{
		return $this->flashMessage($message, Flashes::FLASH_SUCCESS);
	}

	public function flashMessageInfo(string $message): \stdClass
	{
		return $this->flashMessage($message, Flashes::FLASH_INFO);
	}

	public function flashMessageWarning(string $message): \stdClass
	{
		return $this->flashMessage($message, Flashes::FLASH_WARNING);
	}

	public function flashMessageDanger(string $message): \stdClass
	{
		return $this->flashMessage($message, Flashes::FLASH_DANGER);
	}

	/*******************************************************************************************************************
	 *=========================== GET
	 */

	/**
	 * @return Lang
	 */
	public function getLang()
	{
		return $this->lang;
	}

	/**
	 * @return Langs
	 */
	public function getLangsService()
	{
		return $this->langsService;
	}
}
