<?php declare(strict_types = 1);

namespace Core\Model\UI;

use Core\AdminModule\Components\ContentLangSwitcher\ContentLangSwitcher;
use Core\Components\IPartRenderControlFactory;
use Core\Components\PartRenderControl;
use Core\Model\Event\EventDispatcher;
use Core\AdminModule\Components\ContentLangSwitcher\IContentLangSwitcherFactory;
use Core\Model\Event\ControlEvent;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Langs;
use Core\Model\Settings;
use Core\Model\SystemConfig;
use Core\Model\Templating\Template;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Caching\Storage;
use Nette\ComponentModel\IComponent;
use Nette\ComponentModel\IContainer;
use Contributte\Translation\Translator;
use Nette\Application\UI\Control;
use Nette\Caching\Cache;
use Core\Model\Helpers\Strings;

/**
 * @property-read Template                      $template
 * @property-read FrontPresenter|AdminPresenter $presenter
 */
abstract class BaseControl extends Control
{
	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var IContentLangSwitcherFactory @inject */
	public $contentLangSwitcherFactory;

	/** @var IPartRenderControlFactory @inject */
	public $partRenderControlFactory;

	/** @var SystemConfig @inject */
	public $configService;

	/** @var Settings @inject */
	public $settings;

	/** @var Translator */
	protected $translator;

	/** @var EntityManagerDecorator */
	protected $em;

	/** @var Storage */
	protected $cacheStorage;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var Cache */
	protected $cache;

	/** @var Langs */
	protected $langsService;

	/** @var Cache */
	protected $templateFileCache;

	protected array $handleCallbacks = [];

	/** @var BaseWidgetControl[] */
	protected array $widgetControls = [];

	protected function callControlDispatch(string $eventSuffix): void
	{
		if ($this->eventDispatcher) {
			$class      = get_called_class();
			$isOverride = strpos($class, 'Override') !== false;
			$event      = new ControlEvent($this);

			if ($isOverride) {
				$this->eventDispatcher->dispatch($event, str_replace('Override', '', $class) . '::' . $eventSuffix);
			}

			$this->eventDispatcher->dispatch($event, $class . '::' . $eventSuffix);
		}
	}

	public function setParent(?IContainer $parent, string $name = null)
	{
		$parent = parent::setParent($parent, $name);

		$this->monitor(AbstractPresenter::class, function() use ($parent) {
			$this->attached($parent->getPresenter());
		});

		return $parent;
	}

	protected function attached(IComponent $presenter): void
	{
		parent::attached($presenter);

		$this->callControlDispatch('onAttach');
	}

	protected function beforeRender(): void
	{
		$this->callControlDispatch('beforeRender');
	}

	public function handleCallback(string $key, ?array $args = null): void
	{
		$args = func_get_args();
		$c    = $this->handleCallbacks[$key] ?? null;

		if ($c && is_callable($c)) {
			if ($args[1] !== null) {
				$c($args[1]);
			} else {
				$c();
			}
		}
	}

	/**
	 * @param mixed $message
	 * @param mixed $count
	 */
	public function t($message, $count = null, array $parameters = [], ?string $domain = null, ?string $locale = null): string
	{
		return $this->translator->translate($message, $count, $parameters, $domain, $locale);
	}

	/**
	 * Vraci naprimo POST pole od aktualniho presenteru a requestu
	 *
	 * @param string|null $key
	 * @param mixed|null  $default
	 *
	 * @return mixed
	 */
	public function getPost($key = null, $default = null)
	{
		return $this->presenter->getHttpRequest()->getPost($key);
	}

	public function setTranslator(Translator $translator): void { $this->translator = $translator; }

	public function setEntityManager(EntityManagerDecorator $entityManager): void { $this->em = $entityManager; }

	public function setCacheStorage(Storage $cacheStorage): void { $this->cacheStorage = $cacheStorage; }

	public function setImagePipe(ImagePipe $imagePipe): void
	{
		$this->imagePipe  = $imagePipe;
		$this->onAnchor[] = function($control) {
			$control->template->_imagePipe = $this->imagePipe;
		};
	}

	public function setLangsService(Langs $langs): void { $this->langsService = $langs; }

	public function addHandleCallback(string $key, callable $c): self
	{
		$this->handleCallbacks[$key] = $c;

		return $this;
	}

	/**
	 * @param string            $name
	 * @param BaseWidgetControl $control
	 */
	public function addWidgetControl(string $name, BaseWidgetControl $control): void
	{
		$this->widgetControls[$name] = $control;
		$this->addComponent($control, $name);
	}

	/**
	 * @return BaseWidgetControl[]
	 */
	public function getWidgetControls(): array
	{
		return $this->widgetControls;
	}

	/*******************************************************************************************************************
	 * =======================     Getters
	 */

	protected function getTemplateDirs(): array
	{
		$inTemplates = [];
		$inPackages  = [];
		$reflection  = $this->getReflection();
		$namespace   = explode('\\', $reflection->getNamespaceName());

		$paths = [];
		$path  = explode(':', $this->getPresenter()->getName());
		if (count($path) > 2) {
			$tmp     = $path[1];
			$path[1] = $path[0];
			$path[0] = $tmp;
		}

		$paths[] = $path;
		if (count($path) > 2 && $path[1] !== $namespace[0]) {
			$path[1] = $namespace[0];
			$paths[] = $path;
		}

		foreach ($paths as $path) {
			//TODO možnost měnit šablonu
			$oPath  = $path;
			$module = array_shift($path);
			do {
				$pathString = implode(DIRECTORY_SEPARATOR, $path);

				$inTemplates[] = Strings::joinPaths(TEMPLATES_DIR, $module, $this->template->currentSite->getIdent(), $pathString, 'Components');
				$inTemplates[] = Strings::joinPaths(TEMPLATES_DIR, $module, 'default', $pathString, 'Components');
			} while (array_pop($path));

			if ($module === 'Error') {
				$path = $oPath;
				do {
					$pathString = implode(DIRECTORY_SEPARATOR, $path);

					$inTemplates[] = Strings::joinPaths(TEMPLATES_DIR, 'Front', $this->template->currentSite->getIdent(), $pathString, 'Components');
					$inTemplates[] = Strings::joinPaths(TEMPLATES_DIR, 'Front', 'default', $pathString, 'Components');
				} while (array_pop($path));
			}
		}

		foreach ([$reflection, $reflection->getParentClass()] as $ref) {
			if (!$ref instanceof \ReflectionClass) {
				continue;
			}

			$reflectionDir   = dirname((string) $ref->getFileName());
			$tmp             = explode(DIRECTORY_SEPARATOR, $reflectionDir);
			$moduleComponent = [];

			foreach ($tmp as $k => $v) {

				if ($v === 'src') {
					if ($tmp[$k + 1] === 'Model') {
						break 2;
					}

					$moduleComponent[] = $v;
					break;
				}

				$moduleComponent[] = $v;
			}

			$srcPath      = implode(DIRECTORY_SEPARATOR, $moduleComponent);
			$inPackages[] = $srcPath . DIRECTORY_SEPARATOR . $module . 'Module' . str_replace($srcPath, '', $reflectionDir);
			$inPackages[] = $srcPath . DIRECTORY_SEPARATOR . $module . 'Module' . str_replace($srcPath, '', dirname($reflectionDir));
			$inPackages[] = $reflectionDir;
			$inPackages[] = dirname($reflectionDir);
		}

		return [
			'template' => $inTemplates,
			'packages' => $inPackages,
		];
	}

	protected function getTemplateDirsInSrc(): array
	{
		$list          = [];
		$reflectionDir = dirname((string) $this->getReflection()->getFileName());
		if (strpos($reflectionDir, DS . 'custom' . DS) !== false) {
			$reflectionDir = explode(DS . 'custom', $reflectionDir);
			$reflectionDir = SRC_DIR . $reflectionDir[1];
			$list[]        = $reflectionDir;
			$list[]        = dirname($reflectionDir);
		}

		return $list;
	}

	/**
	 * @param string|null $filename
	 * @param string|null $name
	 *
	 * @return string|null
	 * @throws \ReflectionException
	 */
	protected function getTemplateFile($filename = null, $name = null)
	{
		$reflect = new \ReflectionClass($this);

		$name = $name ?: $this->getName();
		if ($filename) {
			$filename = ucfirst($filename);
			$filename = ucfirst($name) . '/' . preg_replace('/Override$/', '', $filename);
		} else {
			$filename = ucfirst($name);
		}

		$result = [
			'packages' => null,
			'template' => null,
		];

		foreach ([$filename, $reflect->getShortName()] as $name) {
			// Hledání šablony v základních složkách
			foreach ($this->getTemplateDirs() as $key => $dirs) {
				if ($result[$key])
					continue;

				foreach ($dirs as $dir) {
					$file = $dir . DS . $name . '.latte';

					if (file_exists($file)) {
						$result[$key] = $file;
						break;
					}
				}
			}

			// Pokud se nenajde, zkontrolovat zanořené v src
			foreach ($this->getTemplateDirsInSrc() as $dir) {
				foreach (glob($dir . '/**/' . $name . '.latte') ?: [] as $f) {
					$result['packages'] = $f;
					break 2;
				}
			}
		}

		$this->template->originTemplate = $result['packages'];

		return $result['template'] ?: $result['packages'];
	}

	protected function createForm(): BaseForm
	{
		$form = new BaseForm();
		$form->setLangsService($this->langsService);
		$form->setTranslator($this->translator);
		$form->setLangsService($this->langsService);

		return $form;
	}

	protected function createGrid(): BaseDataGrid
	{
		$grid               = new BaseDataGrid();
		$grid->langsService = $this->langsService;
		$grid->setTranslator($this->translator);
		$grid->setStrictSessionFilterValues(false);

		return $grid;
	}

	protected function createComponentContentLangSwitcher(): ContentLangSwitcher { return $this->contentLangSwitcherFactory->create(); }

	protected function createComponentPartControl(): PartRenderControl { return $this->partRenderControlFactory->create(); }
}
