<?php declare(strict_types = 1);

namespace Core\Model\UI\DataGrid\Column;

use Core\Model\UI\DataGrid\Traits\TButtonTryAddIcon;
use Nette\Utils\Html;
use Ublaboo;
use Ublaboo\DataGrid\Column\Action\Confirmation\StringConfirmation;
use Ublaboo\DataGrid\DataGrid;
use Ublaboo\DataGrid\Exception\DataGridColumnRendererException;
use Ublaboo\DataGrid\Row;

class Action extends Ublaboo\DataGrid\Column\Action
{
	use TButtonTryAddIcon;

	/** @var string */
	protected        $class  = 'btn btn-xs';
	protected string $bsType = 'secondary';

	public function render(Row $row)
	{
		if (!$this->shouldBeRendered($row)) {
			return null;
		}

		try {
			return $this->useRenderer($row);
		} catch (DataGridColumnRendererException $e) {
		}

		$link = $this->createLink(
			$this->grid,
			$this->href,
			$this->getItemParams($row, $this->params) + $this->parameters
		);

		$a = Html::el('a')->href($link);

		$this->tryAddIcon($a, $this->getIcon($row), $this->getName());

		if ($this->dataAttributes !== []) {
			foreach ($this->dataAttributes as $key => $value) {
				$a->data((string) $key, $value);
			}
		}

		if ($this->attributes !== []) {
			$a->addAttributes($this->attributes);
		}

		if ($this->getName()) {
			$a->addHtml(Html::el('span')->setText($this->translate($this->getName())));
		}

		$title = $this->getTitle($row);

		if ($title !== null) {
			$a->setAttribute('title', $this->translate($title));
		}

		if ($this->class !== null) {
			$a->setAttribute('class', $this->getClass($row));
		}

		$confirmationDialog = $this->getConfirmationDialog($row);

		if ($confirmationDialog !== null) {
			$a->data(static::$dataConfirmAttributeName, $confirmationDialog);
		}

		if ($this->openInNewTab) {
			$a->addAttributes(['target' => '_blank']);
		}

		if ($a instanceof Html) {
			$a->class .= ' btn-' . $this->bsType;
		}

		return $a;
	}

	public function addClass(string $class): self
	{
		$this->class .= ' ' . $class;

		return $this;
	}

	public function setBsType(string $type = 'secondary'): self
	{
		$this->bsType = $type;

		return $this;
	}

	/**
	 * @param string|callable $val
	 */
	public function setConfirm($val): self
	{
		$confirm = null;
		if (is_string($val)) {
			$confirm = new StringConfirmation($val);
		}

		if (is_callable($val)) {
			$confirm = new Ublaboo\DataGrid\Column\Action\Confirmation\CallbackConfirmation($val);
		}

		if ($confirm) {
			$this->setConfirmation($confirm);
		}

		return $this;
	}

	protected function createLink(DataGrid $grid, string $href, array $params): string
	{
		try {
			$link = parent::createLink($grid, $href, $params);
		} catch (\Exception $e) {
			$link = $this->grid->getPresenter()->link($href, $params);
		}

		return $link;
	}

}
