<?php declare(strict_types = 1);

namespace Core\Model\TemplateReader;

use Core\Model\TemplateReader\Entity\ITemplate as EntityITemplate;
use Core\Model\Templating\Macros\TemplateText;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\HiddenInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Nette\Forms\Controls\HiddenField;
use Nette\Forms\IControl;
use Nette\Localization\ITranslator;
use Nette\Utils\FileSystem;
use Nette\Http\Request;
use Nette\Application\UI\ITemplateFactory;
use Core\Model\Images\ImagePipe;

/**
 * Class TemplateReader
 * @package Core\Model\TemplateReader
 */
class TemplateReader
{

	/** @var TemplateTextTypesCollection */
	protected $templateTextTypesCollection;

	/** @var Request */
	protected $httpRequest;

	/** @var ITemplateFactory */
	protected $templateFactory;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var ITranslator */
	protected $translator;

	//TODO typ šablony (místo default proměnná)
	protected $templatesDir;

	/** @var string */
	protected $translateKey = 'templateReader';

	/** @var string */
	protected $tmpDir = TMP_DIR . '/cache/latte/templateReader';

	/** @var array */
	protected $componentStructure;

	/**
	 * TemplateReader constructor.
	 *
	 * @param TemplateTextTypesCollection $templateTextTypesCollection
	 * @param Request                     $request
	 * @param ITemplateFactory            $templateFactory
	 * @param ImagePipe                   $imagePipe
	 */
	public function __construct(TemplateTextTypesCollection $templateTextTypesCollection, Request $request, ITemplateFactory $templateFactory, ImagePipe $imagePipe, ITranslator $translator)
	{
		$this->templateTextTypesCollection = $templateTextTypesCollection;
		$this->httpRequest                 = $request;
		$this->templateFactory             = $templateFactory;
		$this->imagePipe                   = $imagePipe;
		$this->translator                  = $translator;
		$this->reInitComponentStructure();
	}

	/**
	 * Vrátí strukturu komponent v poli pro vykreslení
	 *
	 * @return array
	 */
	public function getComponentStructure()
	{
		return $this->componentStructure;
	}

	/**
	 * Nastavení klíče pro překlad (admin.key.hodnota)
	 *
	 * @param $key
	 */
	public function setTranslateKey($key)
	{
		$this->translateKey = $key;
	}

	/**
	 * Nastaví výchozí stav struktury pro znovuvytvoření
	 */
	private function reInitComponentStructure()
	{
		$this->componentStructure = ['sections' => []];
	}

	/**
	 * Nastavení složky ze kterých se načtou šablony
	 *
	 * @param $dir
	 */
	public function setTemplatesDir($dir)
	{
		$this->templatesDir = $dir;
	}

	/**
	 * Vrátí všechny šablony co najde
	 *
	 * @return array
	 */
	public function getTemplates()
	{

		foreach (glob($this->templatesDir . '/*.latte') as $file) {
			$fileName = pathinfo($file, PATHINFO_FILENAME);
			$tKey     = 'admin.' . $this->translateKey . '.' . $fileName;
			$t        = $this->translator->translate($tKey);

			$templates[basename($file)] = $t == $tKey ? $fileName : $t;
		}

		if (is_array($templates))
			asort($templates, SORT_LOCALE_STRING);

		return $templates ?: [];
	}

	/**
	 * Vrátí složku kam se ukládají temp soubory při vytváření struktury
	 *
	 * @return string
	 */
	protected function getTmpDir()
	{
		if ($this->templatesDir) {
			return $this->tmpDir . DS . basename($this->templatesDir);
		}

		return $this->tmpDir;
	}

	/**
	 * Vrártí složku kde jsou šablony
	 *
	 * @param $template
	 *
	 * @return string
	 */
	public function getTemplatePath($template)
	{
		return $this->templatesDir . DS . $template;
	}

	/**
	 * TODO stejné jako v DataToFormInputs. Rozdil jenom v $this->componentStruture
	 *
	 * Přečte nastavenou šablonu a do komponenty formuláře načte potřebné komponenty
	 *
	 * @param BaseContainer $container
	 * @param string        $file
	 */
	public function loadTemplateComponents(BaseContainer &$container, string $file)
	{
		$this->reInitComponentStructure();
		$sectionNames = [];
		$groupNames   = [];

		foreach ($this->getTemplateInputs($file) as $v) {
			$item = $this->templateTextTypesCollection->getItemByType($v['type'] ?? 'textarea');

			if ($item && $v['name']) {
				if ($v['title']) {
					$v['title'] = $this->translator->translate(...$v['title']);
					$item->setTitle($v['title']);
				}

				$item->setName($v['name']);

				$s = &$this->componentStructure;

				if (isset($v['section'])) {
					if (isset($v['sectionName']))
						$sectionNames[$v['section']] = $this->translator->translate(...(array) $v['sectionName']);
					if (!isset($s['sections'][$v['section']]))
						$s['sections'][$v['section']] = [];
					$s = &$s['sections'][$v['section']];
				}

				if (isset($v['group'])) {
					if (isset($v['groupName']))
						$groupNames[$v['group']] = $this->translator->translate(...(array) $v['groupName']);
					if (!isset($s[$v['group']]))
						$s[$v['group']] = [];
					$s = &$s[$v['group']];
				}

				$s[] = $v['name'];
				$item->loadContainer($container, $v);
			}
		}

		foreach ($this->componentStructure['sections'] as $sk => $sv) {
			if (array_key_exists($sk, $sectionNames)) {
				$this->componentStructure['sections'][$sectionNames[$sk]] = $sv;
				unset($this->componentStructure['sections'][$sk]);
			}
		}

		foreach ($this->componentStructure['sections'] as $sk => $sv) {
			if (is_array($sv)) {
				foreach ($sv as $gk => $gv) {
					if (array_key_exists($gk, $groupNames)) {
						$this->componentStructure['sections'][$sk][$groupNames[$gk]] = $gv;
						unset($this->componentStructure['sections'][$sk][$gk]);
					}
				}
			}
		}
	}

	/**
	 * Nastaví výchozí hodnoty formuláři se zvolenou šablonou
	 *
	 * @param BaseContainer         $container
	 * @param EntityITemplate|array $data
	 * @param null                  $file
	 */
	public function setDefaults(BaseContainer $container, $data, string $template, $file = null)
	{
		/** @var IControl[] $components */
		$components = $container->components;
		if ($data instanceof EntityITemplate === false) {
			$tmp = [];

			foreach ($data as $l => $v) {
				foreach ($components as $name => $c) {
					if (property_exists($c, 'isMultiLanguage') && $c->getIsMultiLanguage()) {
						$tmp[$name][$l] = $v->getText($name) ?? null;
					} else {
						$tmp[$name] = $v->getText($name) ?? null;
					}
				}
			}

			$data = $tmp;
		}

		if ($data instanceof EntityITemplate && $file && $template == $file || !$file) {
			foreach ($components as $k => $c) {
				$value = $data instanceof EntityITemplate ? $data->getText($k) : ($data[$k] ?? null);

				if ($c instanceof SelectInput) {
					if (array_key_exists($value, $c->getItems())) {
						$c->setDefaultValue($value);
					}
				} else if ($c instanceof CheckboxListInput) {
					$tmp = [];
					foreach ($value as $v) {
						if (array_key_exists($v, $c->getItems()))
							$tmp[] = $v;
					}

					$c->setDefaultValue($tmp);
				} else if ($c instanceof HiddenInput) {
					$c->reSetValue($value);
				} else if ($c instanceof \Nette\Forms\Controls\BaseControl) {
					$c->setDefaultValue($value);
				} else if ($c instanceof BaseContainer) {
					$this->setDefaults($c, $value);
				}
			}
		}
	}

	/**
	 * Zpracuje zvolenou šablonu a vrátí pole s připravenýma hodnotama
	 *
	 * @param $template
	 *
	 * @return array
	 */
	public function getTemplateInputs($template)
	{
		$file        = $this->getTemplatePath($template);
		$fileDir     = dirname($file);
		$content     = @file_get_contents($file);
		$tmpFile     = $this->getTmpDir() . DS . basename($file);
		$saveToCache = [];

		if (!$content)
			return [];

		FileSystem::createDir($this->getTmpDir());

		// Vyhledání všech souborů které se načítají
		// TODO možnost dodělat cache
		$re = '/{(include|import) \'(.*)\'}/m';
		preg_match_all($re, $content, $matches, PREG_SET_ORDER, 0);
		foreach ($matches as $match) {
			if (isset($match[2])) {
				$tf = $this->getTmpDir() . DS . ltrim($match[2], '../');
				FileSystem::createDir(dirname($tf));
				$content = str_replace("'" . $match[2] . "'", $tf, $content);

				$saveToCache[$tf] = file_get_contents($fileDir . DS . $match[2]);
			}
		}
		$saveToCache[$tmpFile] = $content;

		// Upravení obsahu pro použití v adminu a uložení do temp
		foreach ($saveToCache as $f => $content) {
			$re      = '/{(templateText|tt) (.*)}/m';
			$content = preg_replace($re, "R$0", $content);
			//			preg_match('/toVar="(.*)"/m', $content, $vars);
			//
			//			foreach ($vars as $var)
			//				$content = "{var $var[1] = null}" . $content;

			if (strpos($content, '{block content}') !== false) {
				//				$re = '/\{block content\}(.*?)\{\/block\}/sm';
				//				preg_match($re, $content, $matches, PREG_OFFSET_CAPTURE, 0);

				//				if (isset($matches[1][0]))
				//					$content = $matches[1][0];

			}
			//			$content = str_replace('{block content}', '', $content);
			//			$content = str_replace('{/block}', '', $content);
			$content = str_replace('R{tt', '{ttR', $content);
			$content = str_replace('R{templateText', '{templateTextR', $content);
			$content = preg_replace("/{link(.*)}/", "#", $content);

			// TODO přidat podporu pro komponenty
			$content = preg_replace("/{control(.*)}/", "#", $content);

			file_put_contents($f, $content);
		}

		// Vytvoření čísté šablony pro čtení
		$latte             = $this->templateFactory->createTemplate();
		$latte->_imagePipe = $this->imagePipe;
		$latte->setFile($tmpFile);
		$content = $latte->renderToString();

		file_put_contents($tmpFile, $content);

		preg_match_all($re, $content, $output, PREG_SET_ORDER, 0);

		$inputs = [];
		foreach ($output as $v) {
			if (!isset($v[2]))
				continue;

			$arr = TemplateText::parseInputAttrs($v[2]);

			if (isset($arr['name']) && isset($arr['title'])) {
				foreach ($arr as $attrKey => $attrValue) {
					// Připravení dat pro překlad
					if (in_array($attrKey, ['title', 'sectionName', 'groupName'])) {
						$tmp = explode(', ', $attrValue, 2);

						if (isset($tmp[1])) {
							$params = [];
							foreach (explode(',', $tmp[1]) as $param) {
								$param = trim($param);

								list($k, $v) = explode('=>', $param);
								$params[$k] = $v;
							}
							$tmp[1] = $params;
						}

						$arr[$attrKey] = $tmp;
					}
				}

				$inputs[] = $arr;
			}
		}

		return $inputs;
	}
}
