<?php declare(strict_types = 1);

namespace Core\Model\UI;

use Contributte\EventDispatcher\EventDispatcher;
use Core\AdminModule\Components\ContentLangSwitcher\ContentLangSwitcher;
use Core\AdminModule\Components\ContentLangSwitcher\IContentLangSwitcherFactory;
use Core\Model\Event\ControlEvent;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Langs;
use Core\Model\SystemConfig;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Kdyby\Doctrine\EntityManager;
use Nette\Caching\Storages\FileStorage;
use Nette\Localization\ITranslator;
use Nette\Application\UI\Control;
use Nette\Application\UI\Presenter;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Nette\FileNotFoundException;
use Nette\Utils\FileSystem;
use Nette\Utils\Reflection;
use Nette\Utils\Strings;

abstract class BaseControl extends Control
{
	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var IContentLangSwitcherFactory @inject */
	public $contentLangSwitcherFactory;

	/** @var SystemConfig @inject */
	public $configService;

	/** @var ITranslator */
	protected $translator;

	/** @var EntityManager */
	protected $em;

	/** @var IStorage */
	protected $cacheStorage;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var Cache */
	protected $cache;

	/** @var Langs */
	protected $langsService;

	/** @var Cache */
	protected $templateFileCache;

	/**
	 * @param bool $throw
	 *
	 * @return Presenter|AbstractPresenter|\Nette\ComponentModel\IComponent|null
	 */
	public function getPresenter($throw = true)
	{
		return $this->lookup(AbstractPresenter::class, $throw);
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		if ($this->eventDispatcher) {
			$class      = get_called_class();
			$isOverride = strpos($class, 'Override') !== false;
			$event      = new ControlEvent($this);

			if ($isOverride)
				$this->eventDispatcher->dispatch($event, str_replace('Override', '', $class) . '::onAttach');
			$this->eventDispatcher->dispatch($event, $class . '::onAttach');
		}
	}

	/*******************************************************************************************************************
	 * =======================     Helpers
	 */
	public function t($message, $count = null, $parameters = [], $domain = null, $locale = null)
	{
		return $this->translator->translate($message, $count, $parameters, $domain, $locale);
	}

	/**
	 * Vraci naprimo POST pole od aktualniho presenteru a requestu
	 *
	 * @param null $key
	 * @param null $default
	 *
	 * @return mixed
	 */
	public function getPost($key = null, $default = null)
	{
		return $this->getPresenter()->getHttpRequest()->getPost($key, $default);
	}

	/*******************************************************************************************************************
	 * =======================     Setters
	 */

	public function setTranslator(ITranslator $translator)
	{
		$this->translator = $translator;
	}

	public function setEntityManager(EntityManager $entityManager)
	{
		$this->em = $entityManager;
	}

	public function setCacheStorage(IStorage $cacheStorage)
	{
		$this->cacheStorage = $cacheStorage;
	}

	public function setImagePipe(ImagePipe $imagePipe)
	{
		$this->imagePipe  = $imagePipe;
		$this->onAnchor[] = function($control) {
			$control->template->_imagePipe = $this->imagePipe;
		};
	}

	public function setLangsService(Langs $langs)
	{
		$this->langsService = $langs;
	}

	/*******************************************************************************************************************
	 * =======================     Getters
	 */

	protected function getTemplateDirs()
	{
		$list = [];

		$path = explode(':', $this->getPresenter()->getName());
		if (count($path) > 2) {
			$tmp     = $path[1];
			$path[1] = $path[0];
			$path[0] = $tmp;
		}

		//TODO možnost měnit šablonu
		$module = array_shift($path);
		do {
			$list[] = TEMPLATES_DIR . '/' . $module . '/default' . ($path ? '/' . implode('/', $path) : '') . '/Components';
			if ($module == 'Error')
				$list[] = TEMPLATES_DIR . '/Front/default' . ($path ? '/' . implode('/', $path) : '') . '/Components';
		} while (array_pop($path));

		$reflection = $this->getReflection();
		foreach ([$reflection, $reflection->getParentClass()] as $ref) {
			$reflectionDir = dirname($ref->getFileName());
			$list[]        = $reflectionDir;
			$list[]        = dirname($reflectionDir);
		}

		$list = array_merge($list, $this->getTemplateDirsInSrc());

		return $list;
	}

	protected function getTemplateDirsInSrc()
	{
		$list          = [];
		$reflectionDir = dirname($this->getReflection()->getFileName());
		if (strpos($reflectionDir, DS . 'custom' . DS) !== false) {
			$reflectionDir = explode(DS . 'custom', $reflectionDir);
			$reflectionDir = SRC_DIR . $reflectionDir[1];
			$list[]        = $reflectionDir;
			$list[]        = dirname($reflectionDir);
		}

		return $list;
	}

	/**
	 * TODO možná ztráta rychlost. Ověřit
	 *
	 * @param null|string $filename
	 * @param null|string $name
	 *
	 * @return string
	 */
	protected function getTemplateFile($filename = null, $name = null)
	{
		$reflect                        = new \ReflectionClass($this);
		$this->template->originTemplate = str_replace('.php', '.latte', $reflect->getFileName());

		$name = $name ?: $this->getName();
		if ($filename) {
			$filename = ucfirst($filename);
			$filename = ucfirst($name) . '/' . preg_replace('/Override$/', '', $filename);
		} else {
			$filename = ucfirst($name);
		}

		$key  = 'controlTemplate/' . Strings::webalize($this->getPresenter()->getName() . '-' . $this->lookupPath() . '-' . $filename);
		$file = $this->getTemplateFileCache()->load($key, function(&$dep) use ($filename, $reflect) {
			$dep = [
				Cache::EXPIRE  => '1 month',
				Cache::SLIDING => true,
			];

			foreach ([$filename, $reflect->getShortName()] as $name) {
				// Hledání šablony v základních složkách
				foreach ($this->getTemplateDirs() as $dir) {
					$file = $dir . '/' . $name . '.latte';

					if (file_exists($file))
						return $file;
				}

				// Pokud se nenajde, zkontrolovat zanořené v src
				foreach ($this->getTemplateDirsInSrc() as $dir) {
					foreach (glob($dir . '/**/' . $name . '.latte') as $f)
						return $f;
				}
			}

			return null;
		});

		return $file ?: null;
	}

	protected function getTemplateFileCache(): Cache
	{
		if ($this->templateFileCache === null) {
			FileSystem::createDir(TMP_DIR . '/controlTemplateFile');
			$this->templateFileCache = new Cache(new FileStorage(TMP_DIR . '/controlTemplateFile'));
		}

		return $this->templateFileCache;
	}

	/*******************************************************************************************************************
	 * =======================     Controls
	 */

	/**
	 * Upřednostňovaný a obecný způsob vytváření Nette Formu
	 * s překladačem
	 * @return BaseForm
	 */
	protected function createForm()
	{
		$form = new BaseForm();
		$form->setTranslator($this->translator);
		$form->setLangsService($this->langsService);

		return $form;
	}

	/**
	 * Upřednostňovaný a obecný způsob vytváření DataGrid
	 * s překladačem
	 * @return BaseDataGrid
	 */
	protected function createGrid()
	{
		$grid = new BaseDataGrid();
		$grid->setTranslator($this->translator);

		if ($dataSource)
			$grid->setDataSource($dataSource);

		return $grid;
	}

	/*******************************************************************************************************************
	 * ========================  Components
	 */

	protected function createComponentContentLangSwitcher()
	{
		$control               = $this->contentLangSwitcherFactory->create();
		$control->templateFile = 'Inline';
		$control->title        = $this->t('gallery.contentLanguage');

		return $control;
	}
}
