const {readdirSync, existsSync, lstatSync} = require('fs');
const path = require('path');

console.clear();

// pshkcms packages folder in vendor
const vendorPackagesFolder = path.resolve(__dirname, '../');
const customPackagesFolder = path.resolve(__dirname, '../../../custom');
let Aliases = {
	AppModules: path.resolve(__dirname, '../../../node_modules/'),
	AdminCore: path.resolve(__dirname, 'assets/admin/src'),
	AdminCoreScss: path.resolve(__dirname, 'assets/admin/src/scss'),
	AdminCoreScssShared: path.resolve(__dirname, 'assets/admin/src/scss/shared'),
	AdminCoreJs: path.resolve(__dirname, 'assets/admin/src/js')
};

const getKey = path => path.split("\\").pop();

// Get all packages
const getPackages = () => {
	const vendorPackages = readdirSync(vendorPackagesFolder)
		.filter(name => {
			if (existsSync(path.join(vendorPackagesFolder, name, 'assets'))) {
				Aliases[name] = path.resolve(__dirname, path.join(vendorPackagesFolder, name, 'assets'));
				return true;
			}
			console.warn("\x1b[33mWARNING: admin module \x1b[32m" + getKey(name) + "\x1b[33m does not have /assets folder and it won't be included to the bundle \x1b[0m");
			return false
		})
		.map(name => path.join(vendorPackagesFolder, name.toLowerCase()));

	const customPackages = readdirSync(customPackagesFolder)
		.filter(name => {
			if (existsSync(path.join(customPackagesFolder, name, 'assets'))) {
				return true;
			}

			console.warn("\x1b[33mWARNING: custom module \x1b[32m" + getKey(name) + "\x1b[33m does not have /assets folder and it won't be included to the bundle \x1b[0m");
			return false
		})
		.map(name => path.join(customPackagesFolder, name));
	return {vendorPackages, customPackages};
};

// Get all installed stylesheets from packages
const getScss = packages => {
	const mappedPackages = [];
	packages.forEach(pack => {
		const stylePath = `${pack}/assets/admin/src/scss/style.scss`;
		if (existsSync(path.resolve(pack, stylePath))) {
			mappedPackages.push(stylePath);
		} else {
			console.warn(`\x1b[33mWARNING: custom module \x1b[32m${getKey(pack)}\x1b[33m has assets folder but there is no scss/style.scss file - IT WONT BE INCLUDED TO STYLE BUNDLE \x1b[0m`)
		}
	});
	return mappedPackages;
}

// Get all js files from installed packages
const getJs = packages => {
	const mappedPackages = [];
	packages.forEach(pack => {
		const indexJsPath = `${pack}/assets/admin/src/js/index.js`;
		if (existsSync(path.resolve(pack, indexJsPath))) {
			mappedPackages.push(indexJsPath);
		} else {
			console.warn(`\x1b[33mWARNING: custom module \x1b[32m${getKey(pack)}\x1b[33m has assets folder but there is no js/index.js file. It adds no general functionality to modules.js bundle \x1b[0m`)
		}
	});
	return mappedPackages;
}


const PACKAGES = getPackages();

// Defining scss files
const adminConfig = {
	entry: {
		login: [path.resolve(__dirname, `${vendorPackagesFolder}/core/assets/admin/src/scss/login.scss`)],
		admin: [
			path.resolve(__dirname, `${vendorPackagesFolder}/core/assets/admin/src/scss/admin.scss`),
			'core-js/stable',
			'regenerator-runtime/runtime'
		],
		//TODO JS
		modules: [...getScss(PACKAGES.vendorPackages)],
	}
};

if (PACKAGES.customPackages.length > 0)
	adminConfig.entry['custom'] = [
		...getScss(PACKAGES.customPackages),
		...getJs(PACKAGES.customPackages,
			'core-js/stable',
			'regenerator-runtime/runtime'
		)
	];

// Načtení ostatních js souborů v balíčků
for (let k in PACKAGES) {
	PACKAGES[k].forEach(baseDir => {
		let assetsDir = path.resolve(baseDir + '/assets/admin/src/js');
		let packageName = path.parse(baseDir).name;
		let jsFiles = readdirSync(assetsDir).filter(file => file !== 'index.js' && path.parse(assetsDir + '/' + file).ext === '.js');

		jsFiles.forEach(file => {
			file = path.resolve(assetsDir + '/' + file);
			let name = path.parse(file).name;

			adminConfig.entry[packageName + '/' + name] = [
				file,
				'core-js/stable',
				'regenerator-runtime/runtime',
			];
		});
	});
}

// Output
let logOutputOfVendorPackages = "";

console.log();
console.log("Admin (vendor) packages loaded for compilation: ");

// Defining js index files
PACKAGES.vendorPackages.forEach((dir) => {
	const key = getKey(dir);
	adminConfig.entry['admin'].push(path.join(dir, 'assets', 'admin', 'src', 'js', 'index.js'));
	logOutputOfVendorPackages += key + ', ';
});

console.log(logOutputOfVendorPackages);
console.log();
let logOutputOfCustomPackages = "";
console.log("Admin (custom) packages loaded for compilation: ");

PACKAGES.customPackages.forEach((dir) => {
	const key = getKey(dir);
	logOutputOfCustomPackages += key + ', ';
});

console.log(logOutputOfCustomPackages);
console.log();


adminConfig.aliases = Aliases;

module.exports = adminConfig;
