<?php declare(strict_types = 1);

namespace Core\Model;

use Core\Model\Entities\SiteText;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Application\BadRequestException;
use Nette\Http\Request;
use Nette\Http\Response;
use Core\Model\Dao;
use Nette\Localization\ITranslator;

class Sites
{
	/** @var EntityManagerDecorator */
	protected $em;

	/** @var Request */
	protected $httpRequest;

	/** @var Response */
	protected $httpResponse;

	/** @var ITranslator */
	protected $translator;

	/** @var Dao\Site[] */
	protected $cSites;

	public function __construct(EntityManagerDecorator $em, Request $request, Response $response, ITranslator $translator)
	{
		$this->em           = $em;
		$this->httpRequest  = $request;
		$this->httpResponse = $response;
		$this->translator   = $translator;
	}

	/**
	 * @return Dao\Site|null
	 */
	public function getCurrentSite(): ?Dao\Site
	{
		foreach ($this->getSites() as $site)
			foreach ($site->getDomains() as $domain)
				if ($domain->getDomain() === $this->httpRequest->getUrl()->getHost()) {
					$site->currentLang = $this->translator->getLocale();

					return $site;
				}

		return null;
	}

	public function validateAndRedirectSite(): void
	{
		if (!$this->getSites() || php_sapi_name() === 'cli')
			return;

		$host      = $this->httpRequest->getUrl()->getHost();
		$hostParts = explode('.', $host, 2);
		$pathParts = explode('/', ltrim($this->httpRequest->getUrl()->getPath(), '/'));
		$urlLang   = strlen($pathParts[0]) === 2 ? $pathParts[0] : null;

		foreach ($this->getSites() as $site) {
			foreach ($site->getDomains() as $domain) {
				if ($domain->getDomain() === $host) {
					if ($urlLang && isset($site->getDomains()[$urlLang])) {
						$domain = $site->getDomains()[$urlLang];

						if ($domain->getDomain() !== $host || $domain->isDefault)
							$this->redirectToHost($domain->getDomain(), null);
					}

					$this->translator->setLocale($domain->getLang());

					return;
				}

				$parts = explode('.', $domain->getDomain(), 2);

				// host je bez www ale doména s
				if ($host === $parts[1])
					$this->redirectToHost($domain->getDomain());

				// host je s www ale doména bez
				if ($hostParts[1] === $domain->getDomain())
					$this->redirectToHost($domain->getDomain());
			}
		}

		throw new BadRequestException('domainNotFound');
	}

	/**
	 * @return Dao\Site[]
	 */
	public function getSites(): array
	{
		if ($this->cSites === null) {
			$this->cSites = [];
			foreach ($this->em->getRepository(SiteText::class)->createQueryBuilder('st')
				         ->addSelect('s.ident, st.domain, st.siteName, s.siteNameSeparator, s.logo, s.email, st.lang, st.isDefault')
				         ->innerJoin('st.site', 's')
				         ->orderBy('st.isDefault', 'DESC')
				         ->getQuery()->getScalarResult() as $row) {

				if (!isset($this->cSites[$row['ident']])) {
					$site        = new Dao\Site($row['ident']);
					$site->logo  = $row['logo'] ?: null;
					$site->email = $row['email'];

					$this->cSites[$row['ident']] = $site;
				} else {
					$site = $this->cSites[$row['ident']];
				}

				$domain                    = new Dao\SiteDomain($site, $row['domain'], $row['lang'], (int) $row['isDefault']);
				$domain->siteName          = $row['siteName'];
				$domain->siteNameSeparator = $row['siteNameSeparator'];
			}
		}

		return $this->cSites;
	}

	protected function redirectToHost(string $host, $path = true): void
	{
		$url = $this->httpRequest->getUrl();
		$url->setHost($host);
		if (is_string($path))
			$url->setPath($path);
		elseif ($path === null)
			$url->setPath(null);

		$this->httpResponse->redirect($url->getAbsoluteUrl());
		exit;
	}
}
