<?php declare(strict_types = 1);

namespace Core\Model\UI\DataGrid;

use Core\Model\UI\DataGrid\Column;
use Core\Model\UI\Form\BaseForm;
use Nette;
use Nette\Application\UI\Form;
use Nette\Forms\Container;
use Ublaboo\DataGrid\Column\Action;
use Ublaboo\DataGrid\Column\ColumnStatus;
use Ublaboo\DataGrid\DataGrid;
use Ublaboo\DataGrid\InlineEdit\InlineAdd;
use Ublaboo\DataGrid\InlineEdit\InlineEdit;

class BaseDataGrid extends DataGrid
{
	/** @var string */
	protected $tableClass = '';

	/** @var string */
	protected $sortableGroup;

	/** @var bool */
	protected $hideTableFooter = false;

	public function __construct(Nette\ComponentModel\IContainer $parent = null, $name = null)
	{
		parent::__construct($parent, $name);
		//		$this->setItemsPerPageList([10, 20, 50, 100, 1000, 'all']);
		$this->setDefaultPerPage(20);
		$this->setRefreshUrl(false);
		$this->setRememberState(true);
		$this->setOuterFilterColumnsCount(4);

		$this->setTemplateFile(__DIR__ . '/templates/datagrid.latte');

		$this->onAnchor[] = function() {
			$this->getTemplate()->newTemplate = __DIR__ . '/templates/datagrid.latte';
		};
	}

	public function render(): void
	{
		if ($this->isTreeView()) {
			$this->template->extendTemplate = ROOT_DIR . '/vendor/ublaboo/datagrid/src/templates/datagrid_tree.latte';
			$this->setTemplateFile(__DIR__ . '/templates/datagrid_tree.latte');
		} else {
			$this->template->extendTemplate = $this->getOriginalTemplateFile();
		}

		$this->template->tableClass      = $this->tableClass;
		$this->template->sortableGroup   = $this->sortableGroup;
		$this->template->hideTableFooter = $this->hideTableFooter;

		parent::render();
	}

	/**
	 * @param array|array|int[]|array|string[] $itemsPerPageList
	 * @return static
	 */
	public function setItemsPerPageList(array $itemsPerPageList, bool $includeAll = true): DataGrid
	{
		$this->itemsPerPageList = $itemsPerPageList ;

		if ($includeAll) {
			$this->itemsPerPageList[] = 'all';
		}

		if (count($this->getItemsPerPageList()) === 1)
			$this->setPagination(false);

		return $this;
	}

	public function getPerPage()
	{
		$itemsPerPageList = $this->getItemsPerPageList();

		$perPage = $this->perPage ?: reset($itemsPerPageList);

		if ($perPage !== 'all' && !in_array((int) $this->perPage, $itemsPerPageList, true)) {
			$perPage = key($itemsPerPageList);
		}

		return $perPage;
	}

	public function addAction($key, $name, $href = null, array $params = null): Action
	{
		$this->addActionCheck($key);

		$href = $href ?: $key;

		if ($params === null) {
			$params = [$this->primaryKey];
		}

		$control = $this->actions[$key] = new Column\Action($this, $key, $href, $name, $params);

		return $control;
	}

	protected function customFilterComponent()
	{
		$form = new BaseForm($this, 'filter');

		$form->setMethod(static::$formMethod);

		$form->setTranslator($this->getTranslator());

		/**
		 * InlineEdit part
		 */
		$inline_edit_container = $form->addContainer('inline_edit');

		if ($this->inlineEdit instanceof InlineEdit) {
			$inline_edit_container->addSubmit('submit', 'ublaboo_datagrid.save')
				->setValidationScope([$inline_edit_container]);
			$inline_edit_container->addSubmit('cancel', 'ublaboo_datagrid.cancel')
				->setValidationScope(null);

			$this->inlineEdit->onControlAdd($inline_edit_container);
			$this->inlineEdit->onControlAfterAdd($inline_edit_container);
		}

		/**
		 * InlineAdd part
		 */
		$inlineAddContainer = $form->addContainer('inline_add');

		if ($this->inlineAdd instanceof InlineAdd) {
			$inlineAddContainer->addSubmit('submit', 'ublaboo_datagrid.save')
				->setValidationScope([$inlineAddContainer]);
			$inlineAddContainer->addSubmit('cancel', 'ublaboo_datagrid.cancel')
				->setValidationScope(null)
				->setAttribute('data-datagrid-cancel-inline-add', true);

			$this->inlineAdd->onControlAdd($inlineAddContainer);
			$this->inlineAdd->onControlAfterAdd($inlineAddContainer);
		}

		/**
		 * ItemDetail form part
		 */
		$itemsDetailForm = $this->getItemDetailForm();

		if ($itemsDetailForm instanceof Container) {
			$form['items_detail_form'] = $itemsDetailForm;
		}

		/**
		 * Filter part
		 */
		$filterContainer = $form->addContainer('filter');

		foreach ($this->filters as $filter) {
			$filter->addToFormContainer($filterContainer);
		}

		if (!$this->hasAutoSubmit()) {
			$filterContainer['submit'] = $this->getFilterSubmitButton();
		}

		/**
		 * Group action part
		 */
		$groupActionContainer = $form->addContainer('group_action');

		if ($this->hasGroupActions()) {
			$this->getGroupActionCollection()->addToFormContainer($groupActionContainer);
		}

		if ($form->isSubmitted() === false) {
			$this->setFilterContainerDefaults($filterContainer, $this->filter);
		}

		/**
		 * Per page part
		 */
		$select = $form->addSelect('perPage', '', $this->getItemsPerPageList())
			->setTranslator(null);

		if ($form->isSubmitted() === false) {
			$select->setValue($this->getPerPage());
		}

		$form->addSubmit('perPage_submit', 'ublaboo_datagrid.perPage_submit')
			->setValidationScope([$select]);

		$form->onSubmit[] = [$this, 'filterSucceeded'];

		return $form;
	}

	public function createComponentFilter(): Form
	{
		return $this->customFilterComponent();
	}

	/*******************************************************************************************************************
	 * ==================  Columns
	 */

	/**
	 * @param string $key
	 * @param string $name
	 * @param null   $column
	 *
	 * @return \Ublaboo\DataGrid\Column\Column|Column\ColumnStatus
	 * @throws \Ublaboo\DataGrid\Exception\DataGridException
	 */
	public function addColumnStatus($key, $name, $column = null): ColumnStatus
	{
		$column = $column ?: $key;

		return $this->addColumn($key, new Column\ColumnStatus($this, $key, $column, $name));
	}

	/**
	 * @param      $key
	 * @param      $name
	 * @param null $column
	 *
	 * @return \Ublaboo\DataGrid\Column\Column|Column\ColumnPosition
	 */
	public function addColumnPosition($key, $name, $column = null)
	{
		$column = $column ?: $key;

		return $this->addColumn($key, new Column\ColumnPosition($this, $key, $column, $name));
	}

	public function addColumnEmail($key, $name, $column = null)
	{
		$column = $this->addColumnText($key, $name, $column);
		$column->setTemplate(__DIR__ . '/templates/column_email.latte', [
			'key' => $column->getColumn(),
		]);

		return $column;
	}

	public function addColumnPhone($key, $name, $column = null, $country = null, bool $showRegion = false)
	{
		$column = $this->addColumnText($key, $name, $column);

		$column->setAlign('right');
		$column->setTemplate(__DIR__ . '/templates/column_phone.latte', [
			'key'        => $column->getColumn(),
			'keyRegion'  => $country,
			'showRegion' => $showRegion,
		]);

		return $column;
	}

	public function addColumnPrice($key, $name, $column = null)
	{
		$column = $column ?: $key;

		return $this->addColumn($key, new Column\ColumnPrice($this, $key, $column, $name));
	}

	/*******************************************************************************************************************
	 * ==================  Filters
	 */

	public function addFilterRadioNestedList($key, $name, array $flatItems, $column = null)
	{
		$column = $column ?: $column;

		$this->addFilterCheck($key);

		return $this->filters[$key] = new Filter\FilterRadioNestedList($this, $key, $name, $flatItems, $column);
	}

	public function addFilterCheckboxNestedList($key, $name, array $flatItems, $column = null)
	{
		$column = $column ?: $column;

		$this->addFilterCheck($key);

		return $this->filters[$key] = new Filter\FilterCheckboxNestedList($this, $key, $name, $flatItems, $column);
	}

	/*******************************************************************************************************************
	 * ==================  Setters
	 */

	public function setTableClass($class)
	{
		$this->tableClass = $class;

		return $this;
	}

	public function setSortableGroup($group)
	{
		$this->sortableGroup = $group;

		return $this;
	}

	public function hideTableFooter(bool $hide = true): self
	{
		$this->hideTableFooter = $hide;

		return $this;
	}
}
