<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Model\Helpers\Arrays;
use Core\Model\UI\Form\Traits\ChoiceInputTrait;
use Core\Model\UI\Form\Traits\TInputExtend;
use Nette\Forms\Controls\ChoiceControl;
use Nette\Forms\Helpers;
use Nette\InvalidArgumentException;
use Nette\Utils\Html;
use Nette\Utils\Strings;

class RadioNestedInput extends ChoiceControl
{
	use TInputExtend;
	use ChoiceInputTrait;

	const VIEW_TYPE_CLASSIC = 'classic';
	const VIEW_TYPE_COMPACT = 'compact';

	/** @var string */
	protected $type = 'radio';

	/**
	 * @var Html
	 */
	private $container;

	/** @var array */
	protected $flat;

	/** @var array */
	protected $tree;

	/** @var string */
	protected $viewType;

	/**
	 * @param string|object
	 * @param array|null $items
	 */
	public function __construct($label = null, array $items = null)
	{
		if ($items) {
			$this->setFlat($items);
			$this->setItems($items);
			$this->createTree();
		}

		parent::__construct($label);
		$this->control->type = $this->type;
		$this->container     = Html::el();
		$this->setOption('type', $this->type);
		$this->viewType = self::VIEW_TYPE_CLASSIC;
	}

	/**
	 * Generates control's HTML element.
	 * @return Html
	 */
	public function getControl(): Html
	{
		// has to run
		parent::getControl();

		$items     = $this->items;
		$container = $this->container;

		// Sestavení inputů
		$inputs = [];
		foreach ($items as $value => $caption) {
			$disabledOption = $this->isValueDisabled($value);

			$inputId = $this->getHtmlId() . '_' . Strings::webalize((string) $value);

			$label = Html::el('label', [
				'for' => $inputId,
			]);

			$input                            = parent::getControl();
			$input->attrs['name']             = $this->getHtmlName();
			$input->attrs['checked']          = $this->isValueSelected($value);
			$input->attrs['disabled']         = $disabledOption;
			$input->attrs['id']               = $inputId;
			$input->attrs['data-nette-rules'] = Helpers::exportRules($this->getRules()) ?: null;
			$input->attrs['value']            = $value;

			$inputs[$value] = [
				'input' => $input,
				'label' => $label,
			];
		}

		// Sestavení nested listu
		$nestedList = Html::el('div class="nested-list type-' . $this->viewType . '"');
		$nestedLoop = function($data, $nestedList) use (&$nestedLoop, $inputs) {
			$listBlock = Html::el('div class=nested-list__block');

			foreach ($data as $row) {
				$listItem     = Html::el('div class=nested-list__item');
				$listItemHead = Html::el('div class=nested-list__item-head');
				$arrows       = Html::el('div class=nested-list__arrow');

				// Zobrazení +-
				if ($row['children']) {
					$listItem->addClass('nested-list__parent');

					$arrows->addHtml(Html::el('i class="fas fa-plus-square"'));
					$arrows->addHtml(Html::el('i class="fas fa-minus-square"'));
				}
				$listItemHead->addHtml($arrows);

				// Přidání radiobuttonu
				$input = $inputs[$row['id']];
				$listItemHead->addHtml(
					Html::el('div class="pretty p-check"')
						->addHtml($input['input'])
						->addHtml(Html::el('div class=state')
							->addHtml(Html::el('i class="icon fas fa-check"'))
							->addHtml($input['label']
								->addHtml(Html::el('span')->setText($row['name'])))
						)
				);

				// Název
				$listItem->addHtml($listItemHead);

				// Podřazené
				if ($row['children'])
					$nestedLoop($row['children'], $listItem);

				$listBlock->addHtml($listItem);
			}

			$nestedList->addHtml($listBlock);
		};

		$nestedLoop($this->getTree(), $nestedList);

		$container->addHtml($nestedList);

		return $container;
	}

	public function getFlat(): array { return $this->flat ?: []; }

	public function getTree(): array { return $this->tree ?: []; }

	public function setFlat(array $items): self
	{
		if (!$this->checkItemsFormat($items))
			throw new InvalidArgumentException("Items isn't in valid format ([[id => 2, parent => 2, name => test]])");
		$this->flat = $items;

		return $this;
	}

	public function setTree(array $tree): self
	{
		$this->tree = $tree;

		return $this;
	}

	public function getItems(): array
	{
		return $this->items ?: [];
	}

	public function setItems(array $items, bool $useKeys = false, string $arrId = 'id', string $arrVal = 'name'): self
	{
		if (is_array(array_values($items)[0])) {
			$tmp = [];
			foreach ($items as $v) {
				$tmp[$v[$arrId]] = $v[$arrVal];
			}

			$items = $tmp;
		}

		if ($useKeys)
			$items = array_combine($items, $items);

		$this->items = $items;

		return $this;
	}

	public function setViewType(string $type): self
	{
		if (in_array($type, [self::VIEW_TYPE_CLASSIC, self::VIEW_TYPE_COMPACT]))
			$this->viewType = $type;

		return $this;
	}

	protected function createTree(): array
	{
		$this->tree = Arrays::buildTree($this->flat, 'parent', 'id', (int) $this->flat[0]['parent']);

		return $this->tree;
	}

	/**
	 * Zkontroluje formát položek
	 *
	 * @param array $items
	 *
	 * @return bool
	 */
	protected function checkItemsFormat(array $items): bool
	{
		$firstRow = array_values($items)[0];

		return isset($firstRow['id'], $firstRow['parent'], $firstRow['name']);
	}

	public function getValue()
	{
		if ($this->getIsMultiLanguage()) {
			foreach ($this->value as $lang => $val) {
				if (!array_key_exists($val, $this->items))
					$this->value[$lang] = null;
			}

			return $this->value;
		}

		return parent::getValue() ?: $this->value;
	}

}
