<?php declare(strict_types = 1);

namespace Core\AdminModule\Components\Cookies;

use Core\AdminModule\Model\Sites;
use Core\Model\CookieAdditionalServices;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;

class ServicesGrid extends BaseControl
{
	protected CookieAdditionalServices $cookieAdditionalServices;
	protected IServiceFormFactory $serviceFormFactory;
	protected Sites $adminSites;

	public function __construct(CookieAdditionalServices $cookieAdditionalServices, IServiceFormFactory $serviceFormFactory, Sites $adminSites)
	{
		$this->cookieAdditionalServices = $cookieAdditionalServices;
		$this->serviceFormFactory = $serviceFormFactory;
		$this->adminSites = $adminSites;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== COMPONENTS
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->cookieAdditionalServices->getAllWithDefaults());

		$grid->addColumnText('name', 'core.servicesGrid.name');

		if (count($this->adminSites->getOptionsForSelect()) > 1) {
			$grid->addColumnText('sites', 'core.servicesGrid.site');
		}
		$grid->addAction('edit', '', 'edit!')->setRenderCondition(fn($arr) => !$arr['isDefault'])->setIcon('edit')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setRenderCondition(fn($arr) => !$arr['isDefault'])->setIcon('times')->setBsType('danger')
			->addClass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	public function createComponentForm(): ServiceForm
	{
		$control = $this->serviceFormFactory->create();
		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this['grid']->reload();
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this['grid']->reload();
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('core.title.addService');
		$this->template->modal      = 'serviceForm';
		$this->redrawControl('modal');
	}

	public function handleEdit($id): void
	{
		$form = $this['form'];

		$service = $this->cookieAdditionalServices->getEr()->findOneBy(['ident' => $id]);
		$form->setService($service);
		$this->template->modalTitle = $service->name;
		$this->template->modal      = 'serviceForm';
		$this->redrawControl('modal');
	}

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->cookieAdditionalServices->removeByIdent($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}
	
}