<?php declare(strict_types = 1);

namespace Core\FrontModule\Presenters;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Components\Navigation\INavigationFactory;
use Core\Components\Navigation\Navigation;
use Core\FrontModule\Components\ContactForm;
use Core\FrontModule\Components\IContactFormFactory;
use Core\FrontModule\Components\StructureData\DataSchema;
use Core\FrontModule\Components\StructureData\IDataSchemaFactory;
use Core\FrontModule\Model\SeoParser;
use Core\Model\Helpers\Strings;
use Core\Model\Templating\Template;
use Core\Model\Sites;
use Navigations\FrontModule\Components\HrefLang;
use Navigations\FrontModule\Components\IHrefLangFactory;
use Navigations\Model\Navigations;
use Nette\ComponentModel\IComponent;
use Nette\Utils\DateTime;
use Users\Model\Entities\User;
use Users\Model\Event\UserEvent;
use Users\Model\UsersActivity;

/**
 * @property-read Template $template
 */
abstract class BasePresenter extends CustomBasePresenter
{
	/** @var Navigations @inject */
	public Navigations $navigationsService;

	/** @var SeoParser @inject */
	public SeoParser $seoParserService;

	/** @var UsersActivity @inject */
	public UsersActivity $usersActivity;

	/** @var Sites @inject */
	public Sites $sitesService;

	private ?DaoNavigationItem   $activeNavigation  = null;
	protected ?DaoNavigationItem $homepage          = null;
	public                       $baseUrl;
	public array                 $seoDataFromEntity = [];
	protected bool               $skipSetInitial    = false;

	/**
	 * @return void
	 */
	protected function startup()
	{
		parent::startup();

		if (!$this->skipSetInitial) {
			$activeNavigation = $this->request->getParameter('activeNavigation');

			if ($activeNavigation instanceof DaoNavigationItem) {
				$this->activeNavigation = $activeNavigation;
			}

			$this->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();
		}

		$this->template->pageClass = '';

		if ($this->user->isLoggedIn()) {
			/** @var User $user */
			$user = $this->user->getIdentity();
			if ($user->lastActivity() && $user->lastActivity()->getTimestamp() >= (new DateTime())->modify('-10 minutes')->getTimestamp()) {
				$this->usersActivity->setLastActivity($this->user->getId());
			}
		}
	}

	/**
	 * @return void
	 */
	public function beforeRender()
	{
		parent::beforeRender();

		/** @var DaoNavigationItem $activeNavigation */
		$activeNavigation            = $this->getParameter('activeNavigation');
		$activeNavigation->isCurrent = true;

		if ($activeNavigation instanceof DaoNavigationItem) {
			$this->activeNavigation = $activeNavigation;

			if (!$this->skipSetInitial) {
				$path   = [];
				$parent = $activeNavigation->parent;
				while ($parent) {
					$path[] = $parent;

					$parent = $parent->parent;
				}

				$path[] = $activeNavigation;

				/** @var DaoNavigationItem[] $breadcrumbs */
				$breadcrumbs = array_merge($this->breadcrumbsBefore, $path, $this->breadcrumbsAfter);

				if (isset($breadcrumbs[0]) && !$breadcrumbs[0]->isHomepage) {
					array_unshift($breadcrumbs, $this->navigationsService->getHomepage());
				}

				$this->template->breadcrumb       = $breadcrumbs;
				$this->template->activeNavigation = $this->activeNavigation ?: null;
				$this->template->pageClass        .= ($activeNavigation ? $activeNavigation->getParam('pageClass') : '') . ' lang-' . $this->template->lang['shortTag'];
			}

			if ($this->seoDataFromEntity) {
				$activeNavigation->setSeo(array_merge((array) $activeNavigation->getSeo(), (array) $this->seoDataFromEntity));
			}

			if (!$this['title']->getTitle()) {
				if ($activeNavigation->isHomepage && !$activeNavigation->getSeo('title')) {
					$this->setTitle('$siteName', true);
				} else if ($activeNavigation->getSeo('title')) {
					$this->setTitle($activeNavigation->getSeo('title'), true);
				} else {
					$this->setTitle($activeNavigation->title, false);
				}
			}

			$metaSeo = ['description', 'robots'];
			foreach ($metaSeo as $v) {
				if (!$activeNavigation->getSeo($v)) {
					continue;
				}

				$this['meta']->setMeta($v, $activeNavigation->getSeo($v));
			}
		}

		if (!$this->skipSetInitial) {
			$this->template->homepage = $this->getHomepage();
		}

		if ($this->systemConfig->get('testMode') === true) {
			$this['meta']->setMeta('robots', 'noindex, nofollow');
		}

		$this->componentsPack['head']['dataSchemaControl'] = $this['dataSchemaControl'];
		$this->componentsPack['head']['hrefLangControl']   = $this['hrefLangControl'];
	}

	public function getActiveNavigation(): ?DaoNavigationItem
	{
		return $this->activeNavigation;
	}

	public function setActiveNavigation(DaoNavigationItem $item): void { $this->activeNavigation = $item; }

	/*******************************************************************************************************************
	 * =============================== Handle
	 */

	public function handleLogout(): void
	{
		$event = new UserEvent($this->getUser()->getIdentity());
		$this->eventDispatcher->dispatch($event, 'user.logout');

		$this->getUser()->logout(true);
		$homepage = $this->getHomepage();
		$this->redirectUrl($homepage ? ($homepage->link ?: '/') : '/');
	}

	/*******************************************************************************************************************
	 * =============================== Components
	 */

	protected function createComponentContactForm(IContactFormFactory $factory): ContactForm
	{
		return $factory->create();
	}

	protected function createComponentNavigation(INavigationFactory $factory): Navigation
	{
		$control = $factory->create();
		$control->setData($this->navigationsService->getPublishedDaoNavigationStructure());

		return $control;
	}

	protected function createComponentDataSchemaControl(IDataSchemaFactory $factory): DataSchema { return $factory->create(); }

	protected function createComponentHrefLangControl(IHrefLangFactory $factory): HrefLang { return $factory->create(); }

	/*******************************************************************************************************************
	 * =============================== GET / SET
	 */

	/**
	 * @param string $title
	 * @param bool   $includeSiteName
	 */
	public function setTitle($title, $includeSiteName = false)
	{
		$this['title']->setTitle($title, $includeSiteName);

		if ($this->isAjax()) {
			$this->payload->pageTitle = $this['title']->getFullTitle();
		}
	}

	public function getHomepage(): DaoNavigationItem
	{
		if ($this->homepage === null) {
			$this->homepage = $this->navigationsService->getHomepage($this->locale) ?? null;
		}

		return $this->homepage;
	}

	protected function createComponent($name): ?IComponent
	{
		$component = parent::createComponent($name);

		if (!$component) {
			$sl = $this->getComponentFactoriesLocator();

			foreach ($sl->findByTag('dynamicComponent') as $k => $v) {
				if ($v !== $name && Strings::lower($v) !== $name) {
					continue;
				}

				$service = $sl->getService($k);

				if (!$service) {
					continue;
				}

				$component = $service->create();
			}
		}

		return $component;
	}

}
