<?php declare(strict_types = 1);

namespace Core\Model\Application;

use Core\Model\Countries;
use Core\Model\Http\Session;
use Core\Model\Sites;
use Core\Model\SystemConfig;
use Currency\Model\Config as CurrencyConfig;
use Currency\Model\Currencies;
use Navigations\Model\NavigationConfig;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\DI\Container;
use Nette\Http\Request;
use Nette\Http\Response;
use Nette\Utils\DateTime;
use Nette\Utils\Json;

class AppState
{
	const CACHE_NAMESPACE = 'appState';

	protected Sites $sites;

	protected Session $session;

	protected Request $request;

	protected Response $response;

	protected Cache $cache;

	protected Countries $countries;

	protected Container $container;

	protected static array $state = [];

	public function __construct(Sites     $sites, Session $session, Request $request, Response $response, Storage $storage,
	                            Countries $countries, Container $container)
	{
		$this->sites     = $sites;
		$this->session   = $session;
		$this->request   = $request;
		$this->response  = $response;
		$this->countries = $countries;
		$this->container = $container;
		$this->cache     = new Cache($storage, self::CACHE_NAMESPACE);
	}

	public function init(): void
	{
		$cookieBarEnabled = $this->container->getParameters()['system']['cookieBar']['enable'];
		$isCookieBarOpen = false;
		if ($cookieBarEnabled) {
			$cookieBarKey = $this->container->getParameters()['system']['cookieBar']['key'];
			$cookieBarValues = $this->request->getCookie($cookieBarKey);

			$allowCookieServices = [];
			$allowCookieSections = [];
			$isCookieBarOpen = $cookieBarValues === null;
			/** @var array{type: string, services: array<string>, files: array<string>} $cookie */
			foreach ($cookieBarValues ? Json::decode($cookieBarValues, Json::FORCE_ARRAY) : [] as $cookie) {
				$allowCookieSections[] = $cookie['type'];
				foreach ($cookie['services'] as $service) {
					$allowCookieServices[] = $service;
				}
			}

			self::setState('allowCookieServices', $allowCookieServices);
			self::setState('allowCookieSections', $allowCookieSections);
		}
		self::setState('isCookieBarOpen', $isCookieBarOpen);

		$domain = $this->sites->getCurrentSite()->getCurrentDomain();
		if (!$this->request->getCookie($this->getCountryCookieName()) && $domain->defaultCountry) {
			$this->setCountry($domain->defaultCountry);
		}

		if ($this->container->hasService('currency.currencies')) {
			/** @var Currencies $currencies */
			$currencies = $this->container->getService('currency.currencies');
			self::setState('currency', $currencies->getCurrent());

			if (CurrencyConfig::load('changeCountry') && count($currencies->getAll()) > 1) {
				$country = CurrencyConfig::load('changeCountry')[$currencies->getCurrent()->getCode()] ?? null;

				if ($country && $c = $this->countries->checkId($country)) {
					$this->setCountry($c);
				}
			}
		}
	}

	protected function getCountryCookieName(): string
	{
		$name   = 'country';
		$suffix = null;
		$domain = $this->sites->getCurrentSite()->getCurrentDomain();
		if (NavigationConfig::load('showDefaultLangInUrl') || !$domain->isDefault)
			$suffix = $domain->getLang();

		return $name . ($suffix ? ucfirst($suffix) : '');
	}

	public function setCountry(string $country): void
	{
		$key = $this->getCountryCookieName();
		self::setState('freshCountrySet', $country);
		$this->response->setCookie($key, $country, (new DateTime())->modify('+1 year'));
	}

	public function getCountry(): ?string
	{
		if (self::getState('freshCountrySet'))
			return self::getState('freshCountrySet');

		return $this->request->getCookie($this->getCountryCookieName()) ?: $this->sites->getCurrentSite()->getCurrentDomain()->defaultCountry;
	}

	public static function setState($key, $value)
	{
		self::$state[$key] = $value;
	}

	public static function getState($key, $default = null)
	{
		return self::$state[$key] ?? $default;
	}
}
