<?php declare(strict_types = 1);

namespace Core\Model\Mailing;

use Latte\Engine;
use Nette\Bridges\ApplicationLatte\Template;
use Nette\Mail\Mailer;
use Nette\Mail\Message;

class MailBuilder
{

	/** @var Mailer */
	public $sender;

	/** @var Message */
	public $message;

	/** @var Template */
	public $template;

	/**
	 * @param Mailer       $mailer
	 * @param Message|NULL $message
	 */
	public function __construct(Mailer $mailer, Message $message = null)
	{
		$this->sender  = $mailer;
		$this->message = $message ?: new Message();
	}

	/**
	 * @param Message $message
	 *
	 * @return $this
	 */
	public function setMessage(Message $message)
	{
		$this->message = $message;

		return $this;
	}

	/**
	 * TEMPLATE API ************************************************************
	 */

	/**
	 * @param Template $template
	 *
	 * @return void
	 */
	public function setTemplate(Template $template)
	{
		$this->template = clone $template;
	}

	/**
	 * @return Template
	 */
	public function getTemplate()
	{
		return $this->template ?: $this->template = new Template(new Engine());
	}

	/**
	 * @param array $parameters
	 *
	 * @return self
	 */
	public function setParameters(array $parameters)
	{
		$this->getTemplate()->setParameters($parameters);

		return $this;
	}

	/**
	 * @param string $file
	 *
	 * @return self
	 */
	public function setTemplateFile($file)
	{
		$this->getTemplate()->setFile($file);

		return $this;
	}

	/**
	 * MESSAGE API *************************************************************
	 */

	/**
	 * @return Message
	 */
	public function getMessage()
	{
		return $this->message;
	}

	/**
	 * @param string $email
	 * @param string $name
	 *
	 * @return self
	 */
	public function addTo($email, $name = null)
	{
		$this->message->addTo($email, $name);

		return $this;
	}

	/**
	 * @param string $email
	 * @param string $name
	 *
	 * @return self
	 */
	public function addBcc($email, $name = null)
	{
		$this->message->addBcc($email, $name);

		return $this;
	}

	/**
	 * @param string $email
	 * @param string $name
	 *
	 * @return self
	 */
	public function addCc($email, $name = null)
	{
		$this->message->addCc($email, $name);

		return $this;
	}

	/**
	 * @param string $subject
	 *
	 * @return self
	 */
	public function setSubject($subject)
	{
		$this->message->setSubject($subject);

		return $this;
	}

	public function addReplyTo(string $email, string $name = null)
	{
		$this->message->addReplyTo($email, $name);

		return $this;
	}

	/**
	 * @param string $from
	 * @param string $fromName
	 *
	 * @return self
	 */
	public function setFrom($from, $fromName = null)
	{
		$this->message->setFrom($from, $fromName);

		return $this;
	}

	/**
	 * @param callable $callback
	 *
	 * @return self
	 */
	public function call(callable $callback)
	{
		$callback($this->message, $this->template);

		return $this;
	}

	/**
	 * SENDER API **************************************************************
	 */

	/**
	 * Build and send message.
	 *
	 * @return void
	 */
	public function send()
	{
		// Create message
		$message = $this->getMessage();

		// Create template
		$template = $this->getTemplate();
		$template->add('_mail', $message);

		$message->setHtmlBody($template->renderToString());

		// Send message
		$this->sender->send($message);
	}

}
