<?php declare(strict_types = 1);

namespace Core\Model\Subscribers;

use Core\Model\Event\TemplateFactoryEvent;
use Core\Model\Event\TTPreload;
use Core\Model\SystemConfig;
use Core\Model\TemplateReader\TemplateReader;
use DynamicModule\FrontModule\Model\Dao\Group;
use DynamicModule\FrontModule\Model\Dao\Member;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class TemplateSubscriber implements EventSubscriberInterface
{
	protected TemplateReader $templateReader;

	protected ?array $webSettings  = null;
	protected ?array $siteSettings = null;

	protected static array $loadedTemplateInputs = [];

	public function __construct(TemplateReader $templateReader)
	{
		$this->templateReader = $templateReader;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'templateFactory::createTemplate' => ['createTemplate', 250],
			'core.ttPreload'                  => 'ttPreload',
		];
	}

	public function createTemplate(TemplateFactoryEvent $event): void
	{
		if ($event->template->webSettings) {
			if ($this->webSettings === null) {
				$this->webSettings = $this->templateReader->readRawData($event->template->webSettings ?: [], SystemConfig::load('globalData'));
			}

			$event->template->webSettings = $this->webSettings;
		}

		if ($event->template->siteSettings && $event->template->currentSite) {
			if ($this->siteSettings === null) {
				$this->siteSettings = $this->templateReader->readRawData(
					$event->template->siteSettings ?: [],
					SystemConfig::load($event->template->currentSite->getIdent() . '_webSettings')
				);
			}

			$event->template->siteSettings = $this->siteSettings;
		}
	}

	public function ttPreload(TTPreload $event): void
	{
		if (!SystemConfig::load('allowProcessTTPreload')) {
			return;
		}

		// Projde polozky dynamnickeho modulu a nastavit do hodnot spravne hodnoty z tt
		$loopDynModule = function(array &$rows) use (&$loopDynModule) {
			foreach ($rows as &$row) {
				$templateInputs = [];
				$data           = [];

				if ($row instanceof Group && $row->module) {
					$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateGroup/', TEMPLATES_DIR, ucfirst($row->module)));
					$templateInputs = $this->templateReader->getTemplateInputs($row->getTemplate());

					foreach ($row->getFields() as $field) {
						if (is_object($field)) {
							$data[$field->getKey()] = $field->getValue();
						}
					}
				} else if ($row instanceof Member && $row->module) {
					$this->templateReader->setTemplatesDir(sprintf('%s/Front/default/%s/TemplateMember/', TEMPLATES_DIR, ucfirst($row->module)));
					$templateInputs = $this->templateReader->getTemplateInputs($row->getTemplate());

					foreach ($row->getFields() as $field) {
						if (is_object($field)) {
							$data[$field->getKey()] = $field->getValue();
						}
					}
				}

				if (!empty($templateInputs) && !empty($data)) {
					$data = $this->templateReader->readRawData($data, $templateInputs);

					if ($row instanceof Member || $row instanceof Group) {
						foreach ($row->getFields() as &$field) {
							if (is_object($field) && isset($data[$field->getKey()])) {
								$field->setValue($data[$field->getKey()]);
							}
						}
					}
				}

				if ($row instanceof Group) {
					if ($row->getChildren()) {
						$loopDynModule($row->getChildren());
					}

					if ($row->getMembers()) {
						$loopDynModule($row->getMembers());
					}
				}
			}
		};

		if ($event->rawData) {
			$loopDynModule($event->rawData);
		}
	}
}
