<?php declare(strict_types = 1);

namespace Core\Model\TemplateTextType;

use Core\Model\Helpers\Arrays;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\TemplateReader\Providers\ITemplateTextType;
use Core\Model\TemplateReader\Providers\TemplateTextType;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\Navigations as FrontNavigation;
use Nette\DI\Container;
use Nette\Forms\Form;
use Nette\Localization\Translator;
use Nette\Utils\Json;

class Link extends TemplateTextType implements ITemplateTextType
{
	/** @var string */
	protected $type = 'link';

	protected Container  $container;
	protected Translator $translator;

	public function __construct(Container $container, Translator $translator)
	{
		$this->container  = $container;
		$this->translator = $translator;
	}

	/**
	 * @param BaseContainer $formContainer
	 * @param array         $params
	 */
	public function loadContainer($formContainer, $params = [], $texts = [])
	{
		$container = $formContainer->addContainer($this->getName(), $this->getTitle());
		$container->addCustomData('multiLang', true);
		$container->addCustomData('renderFile', __DIR__ . '/Link.latte');

		$types = [
			''      => '',
			'text'  => 'admin.ttType.link.typeList.text',
			'phone' => 'admin.ttType.link.typeList.phone',
			'email' => 'admin.ttType.link.typeList.email',
			'file'  => 'admin.ttType.link.typeList.file',
		];

		if ($this->container->hasService('navigations.admin.navigations')) {
			$types['navigation'] = 'admin.ttType.link.typeList.navigation';
		}

		$container->addSelect('type', 'admin.ttType.link.type', $types)
			->getControlPrototype()->setAttribute('data-tt-link-select', $this->getName());
		$container->addBool('openInNewTab', 'admin.ttType.link.openInNewTab')
			->setDefaultValue(0);

		$typesContainer = $container->addContainer('types', 'admin.ttType.link.typesContainer');
		$typesContainer->addCustomData('ttType', $this->getName());
		$typesContainer->addCustomData('multiLang', true);

		$typesContainer->addText('text', 'admin.ttType.link.text');
		$typesContainer->addText('phone', 'admin.ttType.link.phone');
		$typesContainer->addText('email', 'admin.ttType.link.email');
		$typesContainer->addFilesManager('file', 'admin.ttType.link.file');

		if ($params['multiLang'] == true) {
			$typesContainer['text']->setIsMultilanguage();
			$typesContainer['phone']->setIsMultilanguage();
			$typesContainer['email']->setIsMultilanguage();
			$typesContainer['file']->setIsMultilanguage();
		}

		if ($this->container->hasService('navigations.admin.navigations')) {
			/** @var Navigations $navigations */
			$navigations = $this->container->getService('navigations.admin.navigations');

			$typesContainer->addSelect('navigation', 'admin.ttType.link.navigation', ['' => ''] + $navigations->getOptionsForSelect());
		}
	}

	public function render($params)
	{
		$data = $this->getDefault();

		if (is_string($data) && Arrays::isJson($data)) {
			$data = Json::decode($data, Json::FORCE_ARRAY);
		}

		if ($data['type'] === '') {
			return [];
		}

		$lang     = $params['multiLang'] ? $this->translator->getLocale() : null;
		$getValue = static function($v) use ($lang) {
			if (is_array($v) && array_key_exists($lang, $v)) {
				$v = $v[$lang];
			}

			return $v;
		};

		$link   = null;
		$result = [
			'type'   => $data['type'],
			'target' => (int) $data['openInNewTab'] === 1 ? '_blank' : '_self',
			'value'  => $getValue($data['types'][$data['type']]),
		];

		switch ($data['type']) {
			case 'phone':
				$link = 'tel:' . str_replace(' ', '', (string) $getValue($data['types']['phone']));
				break;
			case 'email':
				$link = 'mailto:' . str_replace(' ', '', (string) $getValue($data['types']['email']));
				break;
			case 'navigation':
				if ($this->container->hasService('navigations.navigations')) {
					/** @var FrontNavigation $navigations */
					$navigations = $this->container->getService('navigations.navigations');
					$nav         = $navigations->getNavigation((int) $getValue($data['types']['navigation']));

					if ($nav) {
						$link                 = $nav->link;
						$result['navigation'] = $nav;
					}
				}
				break;
			default:
				$link = $getValue($data['types'][$data['type']]);
		}

		$result['link'] = $link;

		return $result;
	}
}
