<?php declare(strict_types = 1);

namespace Core\Model\UI;

use Contributte\Application\UI\BasePresenter;
use Core\Components\IPartRenderControlFactory;
use Core\Components\JsTranslates\IJsTranslatesFactory;
use Core\Components\JsTranslates\JsTranslates;
use Core\Components\PartRenderControl;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\EventDispatcher;
use Core\AdminModule\Components\ContentLangSwitcher\ContentLangSwitcher;
use Core\AdminModule\Components\ContentLangSwitcher\IContentLangSwitcherFactory;
use Core\Components\Flashes\Flashes;
use Core\Components\Flashes\IFlashesFactory;
use Core\Components\Header\IFaviconFactory;
use Core\Components\Header\IHeaderFactory;
use Core\Components\Header\IMetaFactory;
use Core\Components\Header\ITitleFactory;
use Core\FrontModule\Components\ISearchFormFactory;
use Core\FrontModule\Components\SearchForm;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\DefaultLang;
use Core\Model\Lang\Lang;
use Core\Model\Lang\Langs;
use Core\Model\Module;
use Core\Model\Settings;
use Core\Model\SystemConfig;
use Forms\FrontModule\Components\FormControl;
use Forms\FrontModule\Components\IFormControlFactory;
use Kdyby\Autowired\AutowireComponentFactories;
use Navigations\FrontModule\Components\ILangSwitcherFactory;
use Navigations\FrontModule\Components\LangSwitcher;
use Nette\Application\UI\Multiplier;
use Nette\Localization\ITranslator;
use Symfony\Contracts\EventDispatcher\Event;

abstract class AbstractPresenter extends BasePresenter
{
	use AutowireComponentFactories;
	use TUITemplate;

	/** @persistent */
	public $locale;

	/** @var ITranslator @inject */
	public $translator;

	/** @var ImagePipe @inject */
	public $imagePipe;

	/** @var Langs @inject */
	public $langsService;

	/** @var SystemConfig @inject */
	public $systemConfig;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var Settings|null @inject */
	public ?Settings $settings;

	/** @var Lang */
	protected $lang;

	/** @var string */
	protected $baseUrl;

	protected array $ajaxPayloadPlaceholders = [];

	public function checkRequirements($element): void
	{
		Module::$currentPresenterName = $this->getFullModuleName();
		parent::checkRequirements($element);
	}

	protected function startup()
	{
		parent::startup();
		$this->eventDispatcher->dispatch(new ControlEvent($this), self::class . '::startup');
		$this->dispatchStaticEvent(new ControlEvent($this), 'startup');


		foreach ($this->translator->getLocalesWhitelist() as $l) {
			if ($this->locale && strpos($l, $this->locale) === 0) {
				SetLocale(LC_ALL, $l . ".utf8");
				break;
			}
		}

		$this->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();
		$this->lang    = $this->langsService->getLang($this->translator->getLocale());
	}

	public function beforeRender()
	{
		parent::beforeRender();
		$this->template->title          = $this['title']->getTitle();
		$this->template->fullModuleName = $this->getFullModuleName();
		$this->template->moduleName     = $this->getModuleName();
		$this->template->presenterName  = $this->getName();

		$reflect                        = new \ReflectionClass($this);
		$this->template->originTemplate = dirname($reflect->getFileName()) . '/../Templates/' . str_replace('Presenter.php', '', basename($reflect->getFileName())) . '/' . lcfirst($this->getAction()) . '.latte';

		$this->eventDispatcher->dispatch(new ControlEvent($this), self::class . '::beforeRender');
		$this->dispatchStaticEvent(new ControlEvent($this), 'beforeRender');

		// SEO stránkování
		foreach ($this->getComponents() as $component) {
			if (method_exists($component, 'getPaginator')) {
				$page = $component->getPaginator()->getPage();
				if ($page > 1) {
					$link = $this->link('this', [
						$component->getName() . '-page' => $page - 1,
						'do'                            => $component->getName() . '-paginator',
					]);
					$this['meta']->setOtherHeader('<link rel="prev" href="' . $link . '">');
				}
				if ($page < $component->getPaginator()->getPageCount()) {
					$link = $this->link('this', [
						$component->getName() . '-page' => $page + 1,
						'do'                            => $component->getName() . '-paginator',
					]);
					$this['meta']->setOtherHeader('<link rel="next" href="' . $link . '">');
				}
			}
		}
	}

	protected function tryCall($method, array $params): bool
	{
		$result = parent::tryCall($method, $params);

		if ($result) {
			$this->dispatchStaticEvent(new ControlEvent($this), $method);
		}

		return $result;
	}

	/*******************************************************************************************************************
	 *=========================== Setters
	 */
	protected function setTitle($title, $colonText = null)
	{
		$this['title']->setTitle($title, false);
		$this['title']->setColonText($colonText);

		//TODO remove
		if (isset($this['header']) && empty($this['header']->getTitle()))
			$this->setHeader($title);
	}

	/**
	 * @param string|array $title
	 * @param string|null  $icon
	 * @param string|null  $description
	 */
	protected function setHeader($title = null, $icon = null, $description = null)
	{
		if (is_string($title)) {
			$this['header']->setTitle($this->translator->translate($title));
			$this['title']->setTitle($this->translator->translate($title), false);
		} else if (is_array($title)) {
			$this['header']->setTitle($this->translator->translate(...$title));
			$this['title']->setTitle($this->translator->translate(...$title), false);
		} else
			$this['header']->setTitle($this['title']->getTitle());

		if (!empty($icon))
			$this['header']->setIcon($icon); else $this['header']->setIcon('fas fa-table');

		if (!empty($description))
			$this['header']->setDescription($this->translator->translate($description));
	}

	/*******************************************************************************************************************
	 *=========================== Components
	 */

	protected function createComponentTitle(ITitleFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentFlashes(IFlashesFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentMeta(IMetaFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentFavicon(IFaviconFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentHeader(IHeaderFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentSearchForm(ISearchFormFactory $factory): SearchForm
	{
		return $factory->create();
	}

	protected function createComponentLangSwitcher(ILangSwitcherFactory $factory): LangSwitcher
	{
		return $factory->create();
	}

	protected function createComponentContentLangSwitcher(IContentLangSwitcherFactory $factory): ContentLangSwitcher
	{
		return $factory->create();
	}

	protected function createComponentPartControl(IPartRenderControlFactory $factory): PartRenderControl { return $factory->create(); }

	protected function createComponentFormControl(IFormControlFactory $factory): Multiplier
	{
		return new Multiplier(function(string $formId) use ($factory): FormControl {
			return $factory->create($formId);
		});
	}

	protected function createComponentJsTranslates(IJsTranslatesFactory $factory): JsTranslates { return $factory->create(); }

	/*******************************************************************************************************************
	 *=========================== Helpers
	 */

	protected function dispatchStaticEvent(Event $event, string $eventName): void
	{
		$class = static::class;

		if (strpos($class, 'OverridePresenter') !== false) {
			$this->eventDispatcher->dispatch($event, $class . '::' . $eventName);
			$class = str_replace('OverridePresenter', 'Presenter', $class);
		}

		$this->eventDispatcher->dispatch($event, $class . '::' . $eventName);
	}

	public function getFullModuleName()
	{
		$arr = explode(':', $this->name);
		array_pop($arr);

		return implode(':', $arr);
	}

	public function t($message, $count = null, $parameters = [], $domain = null, $locale = null)
	{
		return $this->translator->translate($message, $count, $parameters, $domain, $locale);
	}

	public function setPayloadAjaxPlaceholder(string $key, $value): void
	{
		$this->ajaxPayloadPlaceholders[$key] = $value;
	}

	public function sendPayload(): void
	{
		$this->payload->ajaxPlaceholders = $this->ajaxPayloadPlaceholders;
		parent::sendPayload();
	}

	/*******************************************************************************************************************
	 *=========================== Flashes
	 */

	public function flashMessageSuccess($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_SUCCESS);
	}

	public function flashMessageInfo($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_INFO);
	}

	public function flashMessageWarning($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_WARNING);
	}

	public function flashMessageDanger($message)
	{
		return parent::flashMessage($message, Flashes::FLASH_DANGER);
	}

	/*******************************************************************************************************************
	 *=========================== GET
	 */

	/**
	 * @return Lang
	 */
	public function getLang()
	{
		return $this->lang;
	}

	/**
	 * @return Langs
	 */
	public function getLangsService()
	{
		return $this->langsService;
	}
}
