<?php declare(strict_types = 1);

namespace Core\Model\UI\Form\Controls;

use Core\Model\UI\Form\Traits\ChoiceInputTrait;
use Core\Model\UI\Form\Traits\TInputExtend;
use Nette\Forms\Controls\ChoiceControl;
use Nette\Utils\Html;

class BoolInput extends ChoiceControl
{
	use ChoiceInputTrait;
	use TInputExtend;

	/** @var array */
	protected $items = [];

	/** @var string */
	protected $textTrue = 'default.yes';

	/** @var string */
	protected $textFalse = 'default.no';

	/** @var Html */
	private $container;

	public function __construct($label = null, array $items)
	{
		parent::__construct($label, []);
		$this->control->type = 'radio';
		$this->container     = Html::el('div', ['class' => 'btn-group bool-switcher-control']);
		$this->setOption('type', 'radio');

		$this->setItems(empty($items) ? $this->getDefaultItems() : $items);
	}

	public function setTextTrue($text)
	{
		$this->textTrue = $text;

		return $this;
	}

	public function setTextFalse($text)
	{
		$this->textFalse = $text;

		return $this;
	}

	public function getDefaultItems()
	{
		$this->items = [
			0 => $this->translate($this->textFalse),
			1 => $this->translate($this->textTrue),
		];

		return $this->items;
	}

	public function getItems()
	{
		$arr = $this->rawItems;

		array_walk($arr, function(&$a) {
			if (is_array($a))
				$a = $this->translate($a['title']);
			else
				$a = $this->translate($a);
		});

		return $arr;
	}

	protected function isValueSelected($value, ?string $lang = null)
	{
		if (is_null($value))
			return false;

		$val = $this->getValue();

		if ($lang) {
			if (isset($val[$lang]))
				return isset($val[$lang]) && $val[$lang] == $value;
			else
				return $value == $val;
		} else {
			return $value == $val;
		}
	}

	public function getValue()
	{
		if ($this->getIsMultiLanguage())
			return $this->value;
		else
			return parent::getValue();
	}

	public function getControl()
	{
		parent::getControl();
		$langs       = $this->getForm()->langsService->getLangs(false);
		$isMultilang = $this->getIsMultiLanguage();

		$items     = $this->getItems();
		$container = $this->container;

		foreach ($isMultilang ? $langs : ['' => null] as $k => $lang) {
			$wrap = Html::el('div', ['data-content-lang' => $k]);

			foreach ($items as $value => $caption) {
				$disabledOption = $this->isValueDisabled($value);
				$isSelected     = $this->isValueSelected($value, $k);

				$rawItem      = $this->rawItems[$value];
				$baseBtnClass = is_array($rawItem) ? $rawItem['btnClass'] : ($value == 1 ? 'success' : 'danger');
				$btnClass     = 'btn-' . ($isSelected ? '' : 'outline-') . $baseBtnClass;

				$label = Html::el('label', [
					'class'          => 'btn ' . $btnClass,
					'data-btn-class' => $baseBtnClass,
				]);
				$input = Html::el('input', [
					'class'    => 'form-check-input',
					'type'     => 'radio',
					'value'    => $value,
					'name'     => $this->getHtmlName() . ($lang ? "[$k]" : ''),
					'id'       => $this->getHtmlId() . ($lang ? "-$k" : ''),
					'checked'  => $isSelected,
					'disabled' => $disabledOption,
				]);

				$label->addHtml($input);
				$label->addText($caption);

				if ($lang)
					$wrap->addHtml($label);
				else
					$container->addHtml($label);
			}

			if ($lang) {
				$container->addHtml($wrap);
			}
		}

		return $container;
	}
}
