<?php declare(strict_types = 1);

namespace Currency\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use Currency\AdminModule\Components\CurrencyForm;
use Currency\AdminModule\Components\ICurrenciesGridFactory;
use Currency\AdminModule\Components\CurrenciesGrid;
use Currency\AdminModule\Components\ICurrencyFormFactory;
use Currency\AdminModule\Model\Security;
use Currency\Model\Entities\Currency;
use Nette\Application\ForbiddenRequestException;
use Nette\Http\IResponse;

class DefaultPresenter extends BasePresenter
{
	/** @var Security @inject */
	public $currencySecurity;

	protected function startup()
	{
		parent::startup();
		$this->setHeader('currency.title.currencies', 'fas fa-coins');
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */
	public function handleAddCurrency(): void
	{
		if (!$this->currencySecurity->new())
			$this->error(null, IResponse::S403_FORBIDDEN);

		$this->template->modalTitle = $this->t('currency.title.addCurrency');
		$this->template->modal      = 'currencyForm';
		$this->redrawControl('modal');
	}

	public function handleEditCurrency($id): void
	{
		if (!$this->currencySecurity->edit())
			$this->error(null, IResponse::S403_FORBIDDEN);

		/** @var Currency $currency */
		$currency = $this->em->getRepository(Currency::class)->find($id);

		if (!$currency)
			$this->error();

		$this['currencyForm']->id = $id;
		$this['currencyForm']->setCurrency((int) $id);
		$this->template->modalTitle = $this->t('currency.title.editCurrency', null, ['currency' => $currency->getCode()]);
		$this->template->modal      = 'currencyForm';
		$this->template->modalDialogClass = 'modal-md';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault()
	{
		if ($this->currencySecurity->new()) {
			$this['navigation']->setData(['header' => [
				[
					'title' => 'currency.menu.addCurrency',
					'link'  => $this->link('addCurrency!'),
					'ico'   => 'plus',
					'class' => 'ajax',
				],
			]]);
		}
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */
	protected function createComponentCurrencyForm(ICurrencyFormFactory $factory): CurrencyForm
	{
		$control = $factory->create();

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[]         = function(BaseForm $form) use ($control) {
				$this->handleEditCurrency((int) $form->getCustomData('currencyId'));
				$this->redrawControl('flashes');
				$this['currenciesGrid']['grid']->reload();
			};
			$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) use ($control) {
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
				$this['currenciesGrid']['grid']->reload();
			};
			$control['form']['saveControl']->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentCurrenciesGrid(ICurrenciesGridFactory $factory): CurrenciesGrid
	{
		return $factory->create();
	}
}
