<?php declare(strict_types = 1);

namespace Currency\FrontModule\Model\Subscribers;

use Core\Model\Event\EventDispatcher;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderCurrency;
use Core\Model\Entities\EntityManagerDecorator;

class OrderFormSubscriber implements EventSubscriberInterface
{
	/** @var EntityManagerDecorator */
	protected $em;

	/** @var Exchange */
	protected $exchangeService;

	/** @var Currencies */
	protected $currenciesService;

	/** @var EventDispatcher */
	protected $eventDispatcher;

	public function __construct(EntityManagerDecorator $em, Exchange $exchange, Currencies $currencies, EventDispatcher $eventDispatcher)
	{
		$this->em                = $em;
		$this->exchangeService   = $exchange;
		$this->currenciesService = $currencies;
		$this->eventDispatcher   = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm' => ['createOrderForm', 100],
			'eshopOrders.orderBeforeSave' => ['orderBeforeSave', 100],
			'eshopOrders.orderOnSuccess'  => ['orderOnSuccess', 255],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$currency = $this->currenciesService->getCurrent();

		$event->form['currency']->setDefaultValue($currency->getCode());
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values   = $event->formData;
		$currency = $this->currenciesService->getAll()[$values->currency] ?? null;

		if ($currency && $currency->getCode() !== $this->currenciesService->getDefaultCode()) {
			$entity           = new OrderCurrency($event->order, $values->currency);
			$entity->rate     = $currency->rate;
			$entity->decimals = $currency->decimals;

			$event->order->currency = $entity;
		}
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		$this->eventDispatcher->dispatch(new OrderEvent($event->order), Order::class . '::postLoad');
	}
}
