<?php declare(strict_types = 1);

namespace Currency\Model;

use h4kuna\Exchange as h4kunaExchange;
use Currency\Model\Driver;
use h4kuna\Exchange\Caching\Cache;
use Nette\Utils\DateTime;

class Exchange
{
	public const DEFAULT = 'default';
	public const CURRENT = 'current';

	/** @var Config */
	protected $config;

	/** @var Cache */
	protected $cache;

	/** @var h4kunaExchange\Exchange */
	protected $exchange;

	/** @var Driver\Database */
	protected $database;

	protected Currencies $currencies;

	public function __construct(Config $config, Driver\Database $database, Currencies $currencies)
	{
		$this->config     = $config;
		$this->database   = $database;
		$this->currencies = $currencies;
		$this->cache      = new Cache(TMP_DIR . '/exchange');
		$this->exchange   = new h4kunaExchange\Exchange($this->cache);

		$this->exchange->setDriver($database);
		$this->exchange->setDefault((string) $config::load('default'));
	}

	public function changeByDate(float $price, \DateTimeInterface $date, string $to = null, ?string $from = null): float
	{
		if (!in_array($to, $this->config->get('whitelist')))
			throw new \InvalidArgumentException("Currency '$to' is forbidden");

		$this->exchange->setDriver($this->database, $date ?: new DateTime());

		return round($this->exchange->change($price, $from, $to), 2);
	}

	public function change(float $price, ?string $to = null, ?string $from = null): float
	{
		foreach (['to', 'from'] as $v) {
			if (${$v} === self::DEFAULT)
				${$v} = $this->currencies->getDefaultCode();
			else if (${$v} === self::CURRENT)
				${$v} = $this->currencies->getCurrent()->getCode();
		}

		if (!$to)
			$to = $this->currencies->getCurrent()->getCode();

		if (!in_array($to, $this->config->get('whitelist')))
			throw new \InvalidArgumentException("Currency '$to' is forbidden");

		$this->exchange->setDriver($this->database, new DateTime());

		try {
			$result = round($this->exchange->change($price, $from, $to), 2);
		} catch (\Exception $e) {
			$this->cache->flushCache($this->database);
			$result = round($this->exchange->change($price, $from, $to), 2);
		}

		return $result;
	}
}
