<?php declare(strict_types = 1);

namespace Currency\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Http\Session;
use Core\Model\Sites;
use Core\Model\Templating\Filters\Price;
use Currency\FrontModule\Model\Subscribers\ProductsSubscriber;
use Currency\Model\Entities\Currency;
use Nette\Caching\Cache;
use Nette\Http\SessionSection;
use Nette\Http\Request as HttpRequest;

class Currencies extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'currencies';

	/** @var string */
	protected $entityClass = Currency::class;

	protected Sites $sitesService;

	protected SessionSection $sessionSection;

	protected HttpRequest $httpRequest;

	public static $currentCodeOverride = null;

	protected $cCurrent, $cAll, $cActive;

	public function __construct(Sites $sites, Session $session, HttpRequest $httpRequest)
	{
		$this->sitesService   = $sites;
		$this->httpRequest    = $httpRequest;
		$this->sessionSection = $session->getSection('currency');
	}

	public function getCache(): Cache
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function getDefaultCode(): string { return (string) Config::load('default'); }

	/**
	 * @return Currency[]
	 */
	public function getActive(): array
	{
		if ($this->cActive === null) {
			$this->cActive = [];
			foreach ($this->getAll() as $currency) {
				if ($currency->isActive)
					$this->cActive[$currency->getCode()] = $currency;
			}
		}

		return $this->cActive;
	}

	/**
	 * @return Currency[]
	 */
	public function getAll(): array
	{
		$siteIdent = $this->sitesService->getCurrentSite()->getIdent();

		try {
			if ($this->cAll[$siteIdent] === null) {
				$this->cAll[$siteIdent] = $this->getCache()->load(self::CACHE_NAMESPACE . '_' . $siteIdent, function(&$dep) {
					$dep  = [
						Cache::EXPIRATION => '1 hour',
					];
					$data = [];

					foreach ($this->getEr()->createQueryBuilder('c')
						         ->where('c.site = :currentSite OR c.site IS NULL')
						         ->setParameter('currentSite', $this->sitesService->getCurrentSite()->getIdent())
						         ->getQuery()
						         ->getResult() as $row) {
						/** @var Currency $row */
						if (isset($list[$row->getCode()]))
							continue;

						$data[$row->getCode()] = $row;
					}

					return $data;
				});
			}
		} catch (\Exception $e) {
			if (php_sapi_name() === 'cli')
				return [];

			throw $e;
		}

		return $this->cAll[$siteIdent] ?? [];
	}

	public function getCurrent(): Currency
	{
		// TODO vychozi podle eshopu
		$defaultFromConfig  = Config::load('frontEndDefault');
		$all                = $this->getAll();
		$defaultFromSession = $this->sessionSection->current;
		$current            = null;

		if (self::$currentCodeOverride)
			self::$currentCodeOverride = strtoupper((string) self::$currentCodeOverride);

		if (self::$currentCodeOverride && isset($all[self::$currentCodeOverride]))
			return $all[self::$currentCodeOverride];

		foreach ($all as $c) {
			if ($c->getCode() == $defaultFromSession || $c->getId() == $defaultFromSession)
				return $c;

			if ($c->getCode() == $defaultFromConfig || $c->getId() == $defaultFromConfig)
				$current = $c;
		}

		return $current ?: array_values($all)[0];
	}

	public function updatePriceFilter(Price $filter): void
	{
		if (strpos((string) $_SERVER['REQUEST_URI'], '/admin') !== 0) {
			$currency = $this->getCurrent();

			$filter->setCurrencyFromLeft((bool) $currency->currencyFromLeft);
			$filter->setCurrency($currency->symbol);
			$filter->setDecimals($currency->decimals);
		}

		foreach ($this->getAll() as $currency) {
			$filter->addOtherCurrency($currency->getCode(), $currency->symbol, (bool) $currency->currencyFromLeft, $currency->decimals);
		}
	}

	public function setDefaultCurrency(): void
	{
		if (!$this->sessionSection->current) {
			$domain     = $this->sitesService->getCurrentSite()->getCurrentDomain();
			$currencies = $this->getActive();

			if ($domain->defaultCurrency && isset($currencies[$domain->defaultCurrency])) {
				$this->setCurrentCurrency($currencies[$domain->defaultCurrency]->getId());
			}
		}
	}

	public function setCurrencyByUrl(): void
	{
		$urlCode = $this->httpRequest->getQuery('curr');

		if ($urlCode) {
			$currency = $this->getAll()[$urlCode] ?? null;

			if ($currency)
				$this->setCurrentCurrency($currency->getId());
		}
	}

	public function setCurrentCurrency(int $id): void
	{
		/** @var Currency $currency */
		$currency = $this->getEr()->find($id);

		if ($currency->site && $currency->site->getIdent() !== $this->sitesService->getCurrentSite()->getIdent())
			return;

		if ($currency)
			$this->sessionSection->current = $id;
	}

	public static function clearConvertedProducts()
	{
		ProductsSubscriber::$used = [];
	}
}
